// Configuración de la API para el frontend
window.API_CONFIG = {
  // Configurar según el entorno
  BASE_URL: window.location.hostname === 'localhost' 
    ? 'http://localhost:3001' 
    : 'https://admin.bbcrono.airr.cl',
  
  // Token JWT almacenado en localStorage
  getAuthToken: () => localStorage.getItem('authToken'),
  
  // Función de login
  login: async (correo, clave) => {
    try {
      const response = await fetch(`${window.API_CONFIG.BASE_URL}/login`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ correo, clave })
      });
      
      const data = await response.json();
      
      if (!data.success) {
        throw new Error(data.message || 'Error en el login');
      }
      
      // Guardar información del login
      window.API_CONFIG.saveLoginInfo(data);
      
      return data;
    } catch (error) {
      console.error('Error en login:', error);
      throw error;
    }
  },
  
  // Obtener información del usuario actual
  getUserInfo: () => {
    try {
      return {
        token: localStorage.getItem('authToken'),
        id_empresa: localStorage.getItem('id_empresa'),
        id_usuario: localStorage.getItem('id_usuario'),
        rol: localStorage.getItem('rol'),
        usuario: JSON.parse(localStorage.getItem('usuario') || '{}'),
        loginTime: localStorage.getItem('loginTime')
      };
    } catch (error) {
      console.error('Error obteniendo información del usuario:', error);
      return null;
    }
  },
  
  // Guardar información de login
  saveLoginInfo: (loginData) => {
    try {
      // Verificar que tenemos los datos necesarios
      if (!loginData || !loginData.token || !loginData.user) {
        console.error('Datos de login incompletos:', loginData);
        return;
      }
      
      const user = loginData.user;
      
      localStorage.setItem('authToken', loginData.token);
      localStorage.setItem('id_empresa', user.id_empresa ? user.id_empresa.toString() : '1');
      localStorage.setItem('id_usuario', user.id ? user.id.toString() : '1');
      localStorage.setItem('rol', user.rol || 'usuario');
      localStorage.setItem('usuario', JSON.stringify(user));
      localStorage.setItem('loginTime', new Date().toISOString());
      
      console.log('✅ Información de login guardada:', {
        id_empresa: user.id_empresa,
        usuario: user.nombre,
        rol: user.rol
      });
    } catch (error) {
      console.error('Error guardando información de login:', error);
    }
  },
  
  // Limpiar información de login
  clearLoginInfo: () => {
    localStorage.removeItem('authToken');
    localStorage.removeItem('id_empresa');
    localStorage.removeItem('id_usuario');
    localStorage.removeItem('rol');
    localStorage.removeItem('usuario');
    localStorage.removeItem('loginTime');
    console.log('🧹 Información de login limpiada');
  },
  
  // Verificar si el usuario está autenticado
  isAuthenticated: () => {
    const token = localStorage.getItem('authToken');
    const loginTime = localStorage.getItem('loginTime');
    
    if (!token || !loginTime) return false;
    
    // Verificar si el token ha expirado (24 horas)
    const loginDate = new Date(loginTime);
    const now = new Date();
    const hoursElapsed = (now - loginDate) / (1000 * 60 * 60);
    
    if (hoursElapsed > 24) {
      window.API_CONFIG.clearLoginInfo();
      return false;
    }
    
    return true;
  },
  
  // Función para hacer peticiones autenticadas
  request: async (endpoint, options = {}) => {
    const token = localStorage.getItem('authToken');
    const url = `${window.API_CONFIG.BASE_URL}${endpoint}`;
    
    const defaultOptions = {
      headers: {
        'Content-Type': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` }),
        ...options.headers
      },
      ...options
    };

    try {
      const response = await fetch(url, defaultOptions);
      
      if (response.status === 401) {
        // Token expirado o inválido
        window.API_CONFIG.clearLoginInfo();
        window.location.href = './signin.html';
        return;
      }
      
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      
      return await response.json();
    } catch (error) {
      console.error('Error en petición API:', error);
      throw error;
    }
  },
  
  // Login específico
  login: async (correo, clave) => {
    try {
      const response = await fetch(`${window.API_CONFIG.BASE_URL}/api/login`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ correo, clave })
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        console.error('Error respuesta servidor:', errorData);
        throw new Error(errorData.message || errorData.error || 'Error en el login');
      }
      
      const loginData = await response.json();
      
      if (loginData.success) {
        window.API_CONFIG.saveLoginInfo(loginData);
        return loginData;
      } else {
        throw new Error(loginData.error || 'Login fallido');
      }
    } catch (error) {
      console.error('Error de login:', error);
      throw error;
    }
  }
};

// Actualizar la función apiRequest en config.js para usar JWT
window.apiRequest = window.API_CONFIG.request;
