import ApexCharts from "apexcharts";

const chart01 = async () => {
  try {
    const idEmpresa = localStorage.getItem('id_empresa');
    const authToken = localStorage.getItem('authToken');
    
    if (!idEmpresa || !authToken) {
      console.warn('No se encontró ID de empresa o token');
      renderFallbackChart();
      return;
    }

    // Usar el nuevo endpoint de ventas mensuales
    const res = await fetch(`http://localhost:3001/api/ventas/mensuales?empresa=${idEmpresa}`, {
      headers: {
        'Authorization': `Bearer ${authToken}`,
        'Content-Type': 'application/json'
      }
    });

    if (!res.ok) {
      throw new Error(`HTTP ${res.status}: ${res.statusText}`);
    }

    const response = await res.json();

    console.log('Datos de ventas mensuales recibidos:', response);

    if (!response.success || !Array.isArray(response.data)) {
      console.warn('Los datos de ventas mensuales no son válidos:', response);
      renderFallbackChart();
      return;
    }

    const ventasMensuales = response.data;
    const añoActual = response.año;

    // Nombres de los meses en español
    const nombresMeses = [
      'Ene', 'Feb', 'Mar', 'Abr', 'May', 'Jun',
      'Jul', 'Ago', 'Sep', 'Oct', 'Nov', 'Dic'
    ];

    // Extraer los datos de ventas por mes
    const data = ventasMensuales.map(venta => venta.total_ventas);
    const labels = nombresMeses;
    
    console.log('Datos del gráfico:', { data, labels, año: añoActual });

    const maxValue = Math.max(...data);
    const dynamicMax = maxValue > 0 ? Math.ceil(maxValue * 1.2) : 10;

    const chartOneOptions = {
      series: [
        {
          name: "Ventas",
          data: data,
        },
      ],
      colors: ["#465fff"],
      chart: {
        fontFamily: "Outfit, sans-serif",
        type: "bar",
        height: 180,
        toolbar: { show: false },
      },
      plotOptions: {
        bar: {
          horizontal: false,
          columnWidth: "39%",
          borderRadius: 5,
          borderRadiusApplication: "end",
        },
      },
      dataLabels: { enabled: false },
      stroke: { show: true, width: 4, colors: ["transparent"] },
      xaxis: {
        categories: labels,
        axisBorder: { show: false },
        axisTicks: { show: false },
      },
      legend: {
        show: true,
        position: "top",
        horizontalAlign: "left",
        fontFamily: "Outfit",
        markers: { radius: 99 },
      },
      yaxis: {
        min: 0,
        max: dynamicMax,
        tickAmount: Math.max(3, Math.min(8, dynamicMax)),
        forceNiceScale: true,
        title: { text: `Ventas ${añoActual}` },
        labels: {
          formatter: val => Math.round(val)
        }
      },
      grid: { yaxis: { lines: { show: true } } },
      fill: { opacity: 1 },
      tooltip: {
        x: { show: false },
        y: { formatter: val => `${val} ventas` },
      },
    };

    const chartSelector = document.querySelectorAll("#chartOne");
    if (chartSelector.length) {
      const chartOne = new ApexCharts(
        document.querySelector("#chartOne"),
        chartOneOptions,
      );
      chartOne.render();
    }

  } catch (error) {
    console.error('Error cargando datos del gráfico:', error);
    renderFallbackChart();
  }
};

// Función para renderizar gráfico de fallback
const renderFallbackChart = () => {
  const fallbackData = [2, 5, 3, 8, 12, 15, 10, 6, 9, 4, 7, 11];
  
  const chartOneOptions = {
    series: [
      {
        name: "Ventas (ejemplo)",
        data: fallbackData,
      },
    ],
    colors: ["#cccccc"],
    chart: {
      fontFamily: "Outfit, sans-serif",
      type: "bar",
      height: 180,
      toolbar: { show: false },
    },
    plotOptions: {
      bar: {
        horizontal: false,
        columnWidth: "39%",
        borderRadius: 5,
        borderRadiusApplication: "end",
      },
    },
    dataLabels: { enabled: false },
    stroke: { show: true, width: 4, colors: ["transparent"] },
    xaxis: {
      categories: ['Ene', 'Feb', 'Mar', 'Abr', 'May', 'Jun', 'Jul', 'Ago', 'Sep', 'Oct', 'Nov', 'Dic'],
      axisBorder: { show: false },
      axisTicks: { show: false },
    },
    legend: {
      show: true,
      position: "top",
      horizontalAlign: "left",
      fontFamily: "Outfit",
      markers: { radius: 99 },
    },
    yaxis: {
      min: 0,
      max: Math.ceil(Math.max(...fallbackData) * 1.2),
      tickAmount: 5,
      forceNiceScale: true,
      title: { text: "Ventas (sin conexión)" },
      labels: {
        formatter: val => Math.round(val)
      }
    },
    grid: { yaxis: { lines: { show: true } } },
    fill: { opacity: 0.6 },
    tooltip: {
      x: { show: false },
      y: { formatter: () => "Datos no disponibles" },
    },
  };

  const chartSelector = document.querySelectorAll("#chartOne");
  if (chartSelector.length) {
    const chartOne = new ApexCharts(
      document.querySelector("#chartOne"),
      chartOneOptions,
    );
    chartOne.render();
  }
};

export default chart01;