import ApexCharts from "apexcharts";
import flatpickr from "flatpickr";

// ===== chartThree - Comparación detallada anual
let chartThreeInstance = null;
let ventasData = null;
let currentFilter = 'overview';

const chart03 = async () => {
  try {
    const idEmpresa = localStorage.getItem('id_empresa');
    const authToken = localStorage.getItem('authToken');
    
    if (!idEmpresa || !authToken) {
      console.warn('No se encontró ID de empresa o token para chart-03');
      renderFallbackChart();
      return;
    }

    // Usar directamente el endpoint básico de ventas que sabemos que funciona
    const resBasico = await fetch(`http://localhost:3001/api/ventas?empresa=${idEmpresa}`, {
      headers: {
        'Authorization': `Bearer ${authToken}`,
        'Content-Type': 'application/json'
      }
    });

    if (!resBasico.ok) {
      throw new Error(`HTTP ${resBasico.status}: ${resBasico.statusText}`);
    }

    const ventasResponse = await resBasico.json();
    
    console.log('🔍 Respuesta del endpoint básico de ventas:', ventasResponse);

    // Verificar si la respuesta tiene el formato {success: true, data: []}
    let ventasBasicas;
    if (ventasResponse.success && Array.isArray(ventasResponse.data)) {
      ventasBasicas = ventasResponse.data;
    } else if (Array.isArray(ventasResponse)) {
      ventasBasicas = ventasResponse;
    } else {
      console.warn('Los datos de ventas no son válidos para chart-03:', ventasResponse);
      renderFallbackChart();
      return;
    }

    console.log('📊 Datos de ventas procesados:', ventasBasicas);

    // Procesar datos básicos para crear estructura de comparación
    ventasData = processBasicVentasData(ventasBasicas);
    
    console.log('📈 Datos procesados para el gráfico:', ventasData);
    
    setupFilterEvents();
    renderChart(currentFilter);

  } catch (error) {
    console.error('Error cargando datos del gráfico 3:', error);
    renderFallbackChart();
  }
};

const processBasicVentasData = (ventas) => {
  const añoActual = new Date().getFullYear();
  const añoAnterior = añoActual - 1;
  
  // Inicializar arrays para ambos años
  const datosAñoActual = Array.from({ length: 12 }, (_, index) => ({
    mes: index + 1,
    total_ventas: 0,
    monto_total: 0,
    precio_promedio: 0
  }));
  
  const datosAñoAnterior = Array.from({ length: 12 }, (_, index) => ({
    mes: index + 1,
    total_ventas: 0,
    monto_total: 0,
    precio_promedio: 0
  }));

  // Procesar ventas por año y mes
  const ventasPorAñoMes = {};
  
  ventas.forEach(venta => {
    if (!venta.fecha_venta) return;
    
    const fecha = new Date(venta.fecha_venta);
    const año = fecha.getFullYear();
    const mes = fecha.getMonth() + 1; // 1-12
    const precio = parseFloat(venta.precio_venta) || 0;
    
    if (año !== añoActual && año !== añoAnterior) return;
    
    const key = `${año}-${mes}`;
    if (!ventasPorAñoMes[key]) {
      ventasPorAñoMes[key] = {
        año,
        mes,
        ventas: [],
        total_ventas: 0,
        monto_total: 0
      };
    }
    
    ventasPorAñoMes[key].ventas.push(precio);
    ventasPorAñoMes[key].total_ventas++;
    ventasPorAñoMes[key].monto_total += precio;
  });
  
  // Llenar los datos procesados
  Object.values(ventasPorAñoMes).forEach(({ año, mes, total_ventas, monto_total, ventas }) => {
    const precio_promedio = ventas.length > 0 ? monto_total / ventas.length : 0;
    const mesIndex = mes - 1;
    
    const datosVenta = {
      mes,
      total_ventas,
      monto_total,
      precio_promedio
    };
    
    if (año === añoActual && mesIndex >= 0 && mesIndex < 12) {
      datosAñoActual[mesIndex] = datosVenta;
    } else if (año === añoAnterior && mesIndex >= 0 && mesIndex < 12) {
      datosAñoAnterior[mesIndex] = datosVenta;
    }
  });
  
  // Calcular totales
  const totalActual = datosAñoActual.reduce((sum, mes) => sum + mes.monto_total, 0);
  const totalAnterior = datosAñoAnterior.reduce((sum, mes) => sum + mes.monto_total, 0);
  const ventasActual = datosAñoActual.reduce((sum, mes) => sum + mes.total_ventas, 0);
  const ventasAnterior = datosAñoAnterior.reduce((sum, mes) => sum + mes.total_ventas, 0);
  
  return {
    success: true,
    año_actual: añoActual,
    año_anterior: añoAnterior,
    mes_actual: new Date().getMonth() + 1,
    datos_año_actual: datosAñoActual,
    datos_año_anterior: datosAñoAnterior,
    totales: {
      año_actual: totalActual,
      año_anterior: totalAnterior,
      ventas_año_actual: ventasActual,
      ventas_año_anterior: ventasAnterior,
      diferencia_total: totalActual - totalAnterior,
      porcentaje_cambio: totalAnterior > 0 ? ((totalActual - totalAnterior) / totalAnterior) * 100 : 0
    }
  };
};

const setupFilterEvents = () => {
  // Configurar botones de filtro
  const overviewBtn = document.getElementById('btn-overview');
  const salesBtn = document.getElementById('btn-sales');
  const revenueBtn = document.getElementById('btn-revenue');
  
  const updateButtonStyles = (activeBtn) => {
    [overviewBtn, salesBtn, revenueBtn].forEach(btn => {
      if (btn) {
        // Limpiar clases existentes
        btn.classList.remove('shadow-theme-xs', 'text-gray-900', 'dark:text-white', 'bg-white', 'dark:bg-gray-800');
        btn.classList.add('text-gray-500', 'dark:text-gray-400');
      }
    });
    
    if (activeBtn) {
      // Activar botón seleccionado
      activeBtn.classList.remove('text-gray-500', 'dark:text-gray-400');
      activeBtn.classList.add('shadow-theme-xs', 'text-gray-900', 'dark:text-white', 'bg-white', 'dark:bg-gray-800');
    }
  };

  // Inicializar estilo del botón overview como activo
  updateButtonStyles(overviewBtn);

  if (overviewBtn) {
    overviewBtn.addEventListener('click', () => {
      currentFilter = 'overview';
      updateButtonStyles(overviewBtn);
      renderChart('overview');
    });
  }
  
  if (salesBtn) {
    salesBtn.addEventListener('click', () => {
      currentFilter = 'sales';
      updateButtonStyles(salesBtn);
      renderChart('sales');
    });
  }
  
  if (revenueBtn) {
    revenueBtn.addEventListener('click', () => {
      currentFilter = 'revenue';
      updateButtonStyles(revenueBtn);
      renderChart('revenue');
    });
  }

  // Configurar filtro de fechas (simplificado para usar filtros anuales)
  const datePicker = document.querySelector('.datepicker');
  if (datePicker) {
    // Verificar si ya existe una instancia de flatpickr
    if (datePicker._flatpickr) {
      datePicker._flatpickr.destroy();
    }
    
    // Crear nueva instancia específica para chart-03
    if (typeof flatpickr !== 'undefined') {
      const fp = flatpickr(datePicker, {
        mode: "single", // Cambiar a modo single para seleccionar año
        dateFormat: "Y",
        placeholder: "Seleccionar año",
        allowInput: true,
        maxDate: "today",
        minDate: "2020-01-01",
        onChange: function(selectedDates, dateStr, instance) {
          console.log('Cambio en datepicker año:', dateStr);
          if (selectedDates.length === 1) {
            const selectedYear = selectedDates[0].getFullYear();
            loadYearComparison(selectedYear);
          }
        },
        onReady: function(selectedDates, dateStr, instance) {
          console.log('Datepicker de año listo para chart-03');
        }
      });
    } else {
      console.warn('Flatpickr no está disponible');
    }
  }
};

const loadYearComparison = async (selectedYear) => {
  try {
    const idEmpresa = localStorage.getItem('id_empresa');
    const authToken = localStorage.getItem('authToken');
    
    if (!idEmpresa || !authToken) return;

    // Usar endpoint básico que sabemos que funciona
    const resBasico = await fetch(`http://localhost:3001/api/ventas?empresa=${idEmpresa}`, {
      headers: {
        'Authorization': `Bearer ${authToken}`,
        'Content-Type': 'application/json'
      }
    });

    if (resBasico.ok) {
      const ventasResponse = await resBasico.json();
      
      // Manejar diferentes formatos de respuesta
      let ventasBasicas;
      if (ventasResponse.success && Array.isArray(ventasResponse.data)) {
        ventasBasicas = ventasResponse.data;
      } else if (Array.isArray(ventasResponse)) {
        ventasBasicas = ventasResponse;
      } else {
        console.warn('Formato de respuesta no válido para año específico');
        return;
      }

      // Filtrar ventas para los años específicos
      const ventasFiltradas = ventasBasicas.filter(venta => {
        if (!venta.fecha_venta) return false;
        const año = new Date(venta.fecha_venta).getFullYear();
        return año === selectedYear || año === (selectedYear - 1);
      });
      
      ventasData = processBasicVentasDataForYears(ventasFiltradas, selectedYear, selectedYear - 1);
      renderChart(currentFilter);
    }
  } catch (error) {
    console.error('Error cargando comparación de año específico:', error);
  }
};

const processBasicVentasDataForYears = (ventas, añoActual, añoAnterior) => {
  // Similar a processBasicVentasData pero para años específicos
  const datosAñoActual = Array.from({ length: 12 }, (_, index) => ({
    mes: index + 1,
    total_ventas: 0,
    monto_total: 0,
    precio_promedio: 0
  }));
  
  const datosAñoAnterior = Array.from({ length: 12 }, (_, index) => ({
    mes: index + 1,
    total_ventas: 0,
    monto_total: 0,
    precio_promedio: 0
  }));

  const ventasPorAñoMes = {};
  
  ventas.forEach(venta => {
    if (!venta.fecha_venta) return;
    
    const fecha = new Date(venta.fecha_venta);
    const año = fecha.getFullYear();
    const mes = fecha.getMonth() + 1;
    const precio = parseFloat(venta.precio_venta) || 0;
    
    if (año !== añoActual && año !== añoAnterior) return;
    
    const key = `${año}-${mes}`;
    if (!ventasPorAñoMes[key]) {
      ventasPorAñoMes[key] = {
        año,
        mes,
        ventas: [],
        total_ventas: 0,
        monto_total: 0
      };
    }
    
    ventasPorAñoMes[key].ventas.push(precio);
    ventasPorAñoMes[key].total_ventas++;
    ventasPorAñoMes[key].monto_total += precio;
  });
  
  Object.values(ventasPorAñoMes).forEach(({ año, mes, total_ventas, monto_total, ventas }) => {
    const precio_promedio = ventas.length > 0 ? monto_total / ventas.length : 0;
    const mesIndex = mes - 1;
    
    const datosVenta = {
      mes,
      total_ventas,
      monto_total,
      precio_promedio
    };
    
    if (año === añoActual && mesIndex >= 0 && mesIndex < 12) {
      datosAñoActual[mesIndex] = datosVenta;
    } else if (año === añoAnterior && mesIndex >= 0 && mesIndex < 12) {
      datosAñoAnterior[mesIndex] = datosVenta;
    }
  });
  
  const totalActual = datosAñoActual.reduce((sum, mes) => sum + mes.monto_total, 0);
  const totalAnterior = datosAñoAnterior.reduce((sum, mes) => sum + mes.monto_total, 0);
  
  return {
    success: true,
    año_actual: añoActual,
    año_anterior: añoAnterior,
    mes_actual: new Date().getMonth() + 1,
    datos_año_actual: datosAñoActual,
    datos_año_anterior: datosAñoAnterior,
    totales: {
      año_actual: totalActual,
      año_anterior: totalAnterior,
      diferencia_total: totalActual - totalAnterior,
      porcentaje_cambio: totalAnterior > 0 ? ((totalActual - totalAnterior) / totalAnterior) * 100 : 0
    }
  };
};

const renderChart = (filterType) => {
  if (!ventasData) {
    renderFallbackChart();
    return;
  }

  const {
    año_actual,
    año_anterior,
    datos_año_actual,
    datos_año_anterior
  } = ventasData;

  console.log('Renderizando gráfico con filtro:', filterType);

  // Extraer datos mensuales
  const salesDataActual = datos_año_actual.map(mes => mes.total_ventas);
  const revenueDataActual = datos_año_actual.map(mes => Math.round(mes.monto_total / 1000000)); // Convertir a millones
  const salesDataAnterior = datos_año_anterior.map(mes => mes.total_ventas);
  const revenueDataAnterior = datos_año_anterior.map(mes => Math.round(mes.monto_total / 1000000));

  // Configurar series según el filtro
  let series = [];
  let colors = [];
  
  switch (filterType) {
    case 'overview':
      series = [
        {
          name: `Ventas ${año_actual}`,
          data: salesDataActual,
        },
        {
          name: `Ganancias ${año_actual} (M)`,
          data: revenueDataActual,
        },
      ];
      colors = ["#465FFF", "#9CB9FF"];
      break;
      
    case 'sales':
      series = [
        {
          name: `Ventas ${año_actual}`,
          data: salesDataActual,
        },
        {
          name: `Ventas ${año_anterior}`,
          data: salesDataAnterior,
        },
      ];
      colors = ["#465FFF", "#9CB9FF"];
      break;
      
    case 'revenue':
      series = [
        {
          name: `Ganancias ${año_actual} (M)`,
          data: revenueDataActual,
        },
        {
          name: `Ganancias ${año_anterior} (M)`,
          data: revenueDataAnterior,
        },
      ];
      colors = ["#10B981", "#6EE7B7"];
      break;
  }

  const chartThreeOptions = {
    series: series,
    legend: {
      show: true,
      position: "top",
      horizontalAlign: "left",
      fontFamily: "Outfit",
      markers: {
        width: 12,
        height: 12,
        radius: 2,
      },
    },
    colors: colors,
    chart: {
      fontFamily: "Outfit, sans-serif",
      height: 310,
      type: "area",
      toolbar: {
        show: false,
      },
    },
    fill: {
      gradient: {
        enabled: true,
        opacityFrom: 0.55,
        opacityTo: 0,
      },
    },
    stroke: {
      curve: "straight",
      width: ["2", "2"],
    },
    markers: {
      size: 0,
    },
    grid: {
      xaxis: {
        lines: {
          show: false,
        },
      },
      yaxis: {
        lines: {
          show: true,
        },
      },
    },
    dataLabels: {
      enabled: false,
    },
    tooltip: {
      x: {
        format: "MMM",
      },
      y: {
        formatter: function (val, opts) {
          if (opts && opts.series && opts.series[opts.seriesIndex] && opts.series[opts.seriesIndex].name) {
            const seriesName = opts.series[opts.seriesIndex].name;
            if (seriesName.includes('Ganancias')) {
              return val + 'M CLP';
            } else if (seriesName.includes('Ventas')) {
              return val + ' ventas';
            }
          }
          return val;
        }
      }
    },
    xaxis: {
      type: "category",
      categories: [
        "Ene", "Feb", "Mar", "Abr", "May", "Jun",
        "Jul", "Ago", "Sep", "Oct", "Nov", "Dic",
      ],
      axisBorder: {
        show: false,
      },
      axisTicks: {
        show: false,
      },
    },
    yaxis: {
      title: {
        style: {
          fontSize: "0px",
        },
      },
    },
  };

  // Limpiar gráfico anterior si existe
  if (chartThreeInstance) {
    chartThreeInstance.destroy();
    chartThreeInstance = null;
  }

  const chartSelector = document.querySelector("#chartThree");
  if (chartSelector) {
    chartSelector.innerHTML = '';
    chartThreeInstance = new ApexCharts(chartSelector, chartThreeOptions);
    chartThreeInstance.render();
  }
};

const renderFallbackChart = () => {
  const fallbackSeries = [
    {
      name: "Datos no disponibles",
      data: [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
    },
  ];

  const chartThreeOptions = {
    series: fallbackSeries,
    legend: {
      show: false,
    },
    colors: ["#cccccc"],
    chart: {
      fontFamily: "Outfit, sans-serif",
      height: 310,
      type: "area",
      toolbar: {
        show: false,
      },
    },
    fill: {
      gradient: {
        enabled: true,
        opacityFrom: 0.3,
        opacityTo: 0,
      },
    },
    stroke: {
      curve: "straight",
      width: ["2"],
    },
    markers: {
      size: 0,
    },
    grid: {
      xaxis: {
        lines: {
          show: false,
        },
      },
      yaxis: {
        lines: {
          show: true,
        },
      },
    },
    dataLabels: {
      enabled: false,
    },
    tooltip: {
      enabled: false,
    },
    xaxis: {
      type: "category",
      categories: [
        "Ene", "Feb", "Mar", "Abr", "May", "Jun",
        "Jul", "Ago", "Sep", "Oct", "Nov", "Dic",
      ],
      axisBorder: {
        show: false,
      },
      axisTicks: {
        show: false,
      },
    },
    yaxis: {
      title: {
        style: {
          fontSize: "0px",
        },
      },
    },
  };

  // Limpiar gráfico anterior si existe
  if (chartThreeInstance) {
    chartThreeInstance.destroy();
    chartThreeInstance = null;
  }

  const chartSelector = document.querySelector("#chartThree");
  if (chartSelector) {
    chartSelector.innerHTML = '';
    chartThreeInstance = new ApexCharts(chartSelector, chartThreeOptions);
    chartThreeInstance.render();
  }
};

export default chart03;
