import jsVectorMap from "jsvectormap";
import "jsvectormap/dist/maps/world.js";

// Coordenadas de las regiones según la nueva base de datos
const regionCoords = {
  1: [-18.4833, -70.3167],  // Arica y Parinacota (XV)
  2: [-20.2167, -70.1333],  // Tarapacá (I)
  3: [-23.6500, -70.4000],  // Antofagasta (II)
  4: [-27.3667, -70.3333],  // Atacama (III)
  5: [-29.9000, -71.2500],  // Coquimbo (IV)
  6: [-33.0500, -71.6167],  // Valparaíso (V)
  7: [-33.4500, -70.6667],  // Metropolitana (RM)
  8: [-34.1667, -70.7333],  // O'Higgins (VI)
  9: [-35.4333, -71.6667],  // Maule (VII)
  10: [-36.6167, -71.8333], // Ñuble (XVI)
  11: [-36.8167, -73.0500], // Biobío (VIII)
  12: [-38.7333, -72.6000], // Araucanía (IX)
  13: [-39.8167, -73.2333], // Los Ríos (XIV)
  14: [-41.4667, -72.9333], // Los Lagos (X)
  15: [-45.5667, -72.0667], // Aysén (XI)
  16: [-53.1667, -70.9333]  // Magallanes (XII)
};

const initMap = () => {
  const mapSelector = document.querySelector("#mapOne");
  if (!mapSelector) return;

  // Limpiar el contenedor antes de inicializar un nuevo mapa
  if (window.mapOne) {
    mapSelector.innerHTML = '';
    window.mapOne = null;
  }

  const map = new jsVectorMap({
      selector: "#mapOne",
      map: "world",
      zoomButtons: true,
      focusOn: {
        regions: ['CL'],
        animate: true
      },
      selectedRegions: ['CL'],
      regionStyle: {
        initial: {
          fontFamily: "Outfit",
          fill: "#D9D9D9",
        },
        hover: {
          fillOpacity: 1,
          fill: "#465fff",
        },
        selected: {
          fill: "#465fff"
        }
      },
      markers: [
        {
          name: "Chile",
          coords: [-33.4500, -70.6667], // Santiago (punto central)
        }
      ],
      markerStyle: {
        initial: {
          strokeWidth: 1,
          fill: "#465fff",
          fillOpacity: 1,
          r: 6,
        },
        hover: {
          fill: "#465fff",
          fillOpacity: 1,
        },
        selected: {},
        selectedHover: {},
      },
      focusOn: {
        regions: ['CL'],
        scale: 4,
        animate: true,
      },
      onRegionTooltipShow: function (tooltip, code) {
        if (code === "CL") {
          tooltip.selector.innerHTML = "Chile";
        }
      },
      onMarkerTooltipShow: function (event, index) {
        if (event && event.html && this.markers[index]) {
          const marker = this.markers[index];
          event.html(`
            <div class="map-tooltip">
              <strong>${marker.name || 'Región'}</strong>
              ${marker.total ? `<br>${marker.total} clientes` : ''}
            </div>
          `);
        }
      },
      zoomMin: 3,
      zoomMax: 8,
    });

    // Guardamos la referencia del mapa
    window.mapOne = map;
  };

const loadMapData = async () => {
  try {
    const idEmpresa = localStorage.getItem('id_empresa');
    const authToken = localStorage.getItem('authToken');

    if (!idEmpresa || !authToken) {
      console.warn('No se encontró ID de empresa o token para el mapa');
      showMapMessage('No hay datos de empresa disponibles');
      return;
    }

    const url = `${window.API_CONFIG.BASE_URL}/api/clientes?empresa=${idEmpresa}`;
    const response = await fetch(url, {
      headers: {
        'Authorization': `Bearer ${authToken}`,
        'Accept': 'application/json',
        'Content-Type': 'application/json'
      }
    });

    if (!response.ok) {
      throw new Error('Error en la respuesta del servidor');
    }

    const data = await response.json();

    if (!data.success) {
      throw new Error(data.message || 'Error al obtener clientes');
    }

    let clientes = Array.isArray(data.data) ? data.data : [data.data];

    // Objeto para almacenar los contadores por región
    const regiones = {};
    let clientesProcesados = 0;
    let clientesSinRegion = 0;
    
    clientes.forEach((cliente, index) => {
      // La región viene como objeto JSON desde el backend
      let regionData = null;
      let regionId = null;
      
      if (cliente.region && typeof cliente.region === 'string') {
        try {
          regionData = JSON.parse(cliente.region);
          if (regionData && regionData.id) {
            regionId = regionData.id;
          }
        } catch (e) {
          console.error('Error parseando región JSON:', e);
        }
      } else if (cliente.region && typeof cliente.region === 'object') {
        regionData = cliente.region;
        if (regionData && regionData.id) {
          regionId = regionData.id;
        }
      }
      
      // Si no tenemos región desde el JOIN, usar id_region directamente
      if (!regionId && cliente.id_region) {
        regionId = cliente.id_region;
      }
      
      if (!cliente || !regionId) {
        clientesSinRegion++;
        return;
      }
      
      if (!regiones[regionId]) {
        regiones[regionId] = {
          total: 0,
          coords: regionCoords[regionId],
          nombre: regionData ? regionData.nombre : `Región ${regionId}`,
          codigo: regionData ? regionData.codigo : null
        };
      }
      regiones[regionId].total++;
      clientesProcesados++;
    });

    const total = Object.values(regiones).reduce((a, b) => a + b.total, 0);

    // Inicializar el mapa
    initMap();

    // Mostrar las estadísticas de regiones
    const container = document.getElementById('regiones-clientes');
    if (container) {
      container.innerHTML = '';
      
      // Ordenar las regiones por número de clientes (descendente)
      const regionesOrdenadas = Object.entries(regiones)
        .sort(([, a], [, b]) => b.total - a.total);

      regionesOrdenadas.forEach(([regionId, data]) => {
        const porcentaje = total ? Math.round((data.total / total) * 100) : 0;
        container.innerHTML += `
          <div class="flex items-center justify-between mb-4">
            <div>
              <p class="text-theme-sm font-semibold text-gray-800 dark:text-white/90">${data.nombre}</p>
              <span class="block text-theme-xs text-gray-500 dark:text-gray-400">${data.total} clientes</span>
            </div>
            <div class="flex w-full max-w-[140px] items-center gap-3">
              <div class="relative block h-2 w-full max-w-[100px] rounded-sm bg-gray-200 dark:bg-gray-800">
                <div class="absolute left-0 top-0 flex h-full" style="width:${porcentaje}%;">
                  <span class="rounded-sm bg-brand-500 text-xs font-medium text-white" style="width:100%;"></span>
                </div>
              </div>
              <p class="text-theme-sm font-medium text-gray-800 dark:text-white/90">${porcentaje}%</p>
            </div>
          </div>
        `;
      });
    } else {
      console.error('No se encontró el contenedor regiones-clientes');
    }
  } catch (error) {
    console.error('Error cargando datos del mapa:', error);
    showMapMessage('No se pudieron cargar los datos del mapa');
  }
};

const showMapMessage = (message) => {
  const container = document.querySelector('#mapOne');
  if (container) {
    container.innerHTML = `
      <div class="text-center py-8">
        <p class="text-gray-500 dark:text-gray-400">${message}</p>
        <p class="text-xs text-gray-400 mt-2">Inicia sesión para ver los datos</p>
      </div>
    `;
  }
};

const map01 = () => {
  try {
    const mapSelector = document.querySelector("#mapOne");
    if (mapSelector) {
      mapSelector.innerHTML = '';
      if (window.mapOne) {
        window.mapOne = null;
      }
    }

    initMap();

    setTimeout(() => {
      loadMapData();
    }, 100);
  } catch (error) {
    console.error('Error al inicializar el mapa:', error);
    showMapMessage('Error al inicializar el mapa');
  }
};

window.addEventListener('beforeunload', () => {
  const mapSelector = document.querySelector("#mapOne");
  if (mapSelector && window.mapOne) {
    mapSelector.innerHTML = '';
    window.mapOne = null;
  }
});

export default map01;