// table-motos.js - Manejo de tabla dinámica de motos con sev_adm
class TableMotos {
    constructor() {
        this.baseURL = window.API_CONFIG ? window.API_CONFIG.BASE_URL : 'http://localhost:3001';
        this.currentFilters = {};
        this.motos = [];
        this.init();
    }

    init() {
        console.log('🚀 Inicializando TableMotos...');
        this.setupEventListeners();
        this.loadMotos();
    }

    setupEventListeners() {
        // Botón filtrar
        const filterBtn = document.querySelector('[data-action="filter"]');
        if (filterBtn) {
            filterBtn.addEventListener('click', () => this.showFilterModal());
        }

        // Botón ver todas
        const viewAllBtn = document.querySelector('[data-action="view-all"]');
        if (viewAllBtn) {
            viewAllBtn.addEventListener('click', () => this.clearFilters());
        }

        // Búsqueda en tiempo real
        const searchInput = document.querySelector('[data-search="motos"]');
        if (searchInput) {
            searchInput.addEventListener('input', (e) => {
                this.debounce(() => this.filterBySearch(e.target.value), 300);
            });
        }
    }

    async loadMotos(filters = {}) {
        try {
            console.log('🔄 Cargando motos...', filters);
            
            // Verificar autenticación
            if (!window.API_CONFIG || !window.API_CONFIG.isAuthenticated()) {
                console.error('❌ Usuario no autenticado');
                this.showError('Sesión expirada. Por favor, inicie sesión nuevamente.');
                return;
            }

            const token = window.API_CONFIG.getAuthToken();
            if (!token) {
                console.error('❌ Token no encontrado');
                this.showError('Token de autenticación no encontrado');
                return;
            }

            // Construir parámetros de consulta
            const params = new URLSearchParams({
                limite: 50,
                orden: 'fecha_ingreso DESC',
                ...filters
            });

            console.log('📡 Haciendo petición a:', `${this.baseURL}/api/motos?${params}`);

            const response = await fetch(`${this.baseURL}/api/motos?${params}`, {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json'
                }
            });

            console.log('📡 Respuesta recibida:', response.status, response.statusText);

            if (!response.ok) {
                if (response.status === 401) {
                    window.API_CONFIG.clearLoginInfo();
                    this.showError('Sesión expirada. Redirigiendo al login...');
                    setTimeout(() => window.location.href = './signin.html', 2000);
                    return;
                }
                throw new Error(`Error ${response.status}: ${response.statusText}`);
            }

            const data = await response.json();
            console.log('✅ Datos recibidos:', data);
            
            if (data.success) {
                this.motos = data.data || [];
                console.log(`📊 Motos cargadas: ${this.motos.length}`);
                this.renderTable();
                this.updateStats();
            } else {
                console.error('❌ Error en la respuesta:', data.error);
                this.showError(data.error || 'Error al cargar las motos');
            }

        } catch (error) {
            console.error('❌ Error cargando motos:', error);
            this.showError(`No se pudieron cargar las motos: ${error.message}`);
        }
    }

    renderTable() {
        const tbody = document.querySelector('table tbody');
        if (!tbody) {
            console.error('No se encontró el tbody de la tabla');
            return;
        }

        if (this.motos.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="4" class="py-8 text-center">
                        <div class="text-gray-500 dark:text-gray-400">
                            <p class="text-lg mb-2">No se encontraron motos</p>
                            <p class="text-sm">Intente ajustar los filtros de búsqueda</p>
                        </div>
                    </td>
                </tr>
            `;
            return;
        }

        tbody.innerHTML = this.motos.map(moto => `
            <tr class="hover:bg-gray-50 dark:hover:bg-white/[0.02] transition-colors">
                <td class="py-3">
                    <div class="flex items-center">
                        <div class="flex items-center gap-3">
                            <div class="h-[50px] w-[50px] overflow-hidden rounded-md bg-gray-100 dark:bg-gray-800">
                                <img 
                                    src="${moto.imagen_url}" 
                                    alt="${moto.marca} ${moto.modelo}" 
                                    class="h-full w-full object-cover"
                                    onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"
                                />
                                <div class="h-full w-full bg-gray-200 dark:bg-gray-700 hidden items-center justify-center text-2xl" style="display: none;">
                                    🏍️
                                </div>
                            </div>
                            <div>
                                <p class="font-medium text-gray-800 text-theme-sm dark:text-white/90">
                                    ${moto.marca} ${moto.modelo}
                                </p>
                                <span class="text-gray-500 text-theme-xs dark:text-gray-400">
                                    ${moto.año || moto.anio} • ${moto.color}
                                </span>
                            </div>
                        </div>
                    </div>
                </td>
                <td class="py-3">
                    <div class="flex items-center">
                        <div>
                            <p class="text-gray-800 text-theme-sm font-medium dark:text-white/90">
                                ${moto.marca}
                            </p>
                            ${moto.nombre_completo_cliente ? `
                                <span class="text-gray-500 text-theme-xs dark:text-gray-400">
                                    Cliente: ${moto.nombre_completo_cliente}
                                </span>
                            ` : ''}
                        </div>
                    </div>
                </td>
                <td class="py-3">
                    <div class="flex items-center">
                        <div>
                            <p class="text-gray-800 text-theme-sm font-medium dark:text-white/90">
                                ${moto.precio_formateado}
                            </p>
                            ${moto.precio_venta_formateado ? `
                                <span class="text-green-600 text-theme-xs dark:text-green-400">
                                    Vendida: ${moto.precio_venta_formateado}
                                </span>
                            ` : ''}
                        </div>
                    </div>
                </td>
                <td class="py-3">
                    <div class="flex items-center">
                        ${this.getEstadoBadge(moto.estado, moto.origen)}
                        ${(moto.kilometraje || moto.kms) ? `
                            <span class="ml-2 text-gray-500 text-theme-xs dark:text-gray-400">
                                ${this.formatKilometraje(moto.kilometraje || moto.kms)}
                            </span>
                        ` : ''}
                    </div>
                </td>
            </tr>
        `).join('');
    }

    getEstadoBadge(estado, origen) {
        // Estados de la base de datos sev_adm
        const badges = {
            'stock': {
                class: 'bg-success-50 text-success-600 dark:bg-success-500/15 dark:text-success-500',
                text: 'En Stock'
            },
            'reservada': {
                class: 'bg-warning-50 text-warning-600 dark:bg-warning-500/15 dark:text-orange-400',
                text: 'Reservada'
            },
            'vendida': {
                class: 'bg-blue-50 text-blue-600 dark:bg-blue-500/15 dark:text-blue-400',
                text: 'Vendida'
            },
            'mantenimiento': {
                class: 'bg-error-50 text-error-600 dark:bg-error-500/15 dark:text-error-500',
                text: 'Mantenimiento'
            },
            // Estados alternativos para compatibilidad
            'disponible': {
                class: 'bg-success-50 text-success-600 dark:bg-success-500/15 dark:text-success-500',
                text: 'Disponible'
            }
        };

        const origenBadges = {
            'consignacion': '📋',
            'propia': '🏪',
            'parte_pago': '🔄'
        };

        const badge = badges[estado] || badges['stock'];
        const origenIcon = origenBadges[origen] || '';

        return `
            <div class="flex items-center gap-2">
                <p class="rounded-full px-2 py-0.5 text-theme-xs font-medium ${badge.class}">
                    ${origenIcon} ${badge.text}
                </p>
            </div>
        `;
    }

    formatKilometraje(km) {
        if (km >= 1000) {
            return `${(km / 1000).toFixed(1)}k km`;
        }
        return `${km} km`;
    }

    showFilterModal() {
        // Crear modal de filtros dinámico
        const modal = document.createElement('div');
        modal.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
        modal.innerHTML = `
            <div class="bg-white dark:bg-gray-800 rounded-lg p-6 w-full max-w-md mx-4">
                <h3 class="text-lg font-semibold text-gray-800 dark:text-white mb-4">
                    Filtrar Motos
                </h3>
                <form id="filter-form" class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                            Marca
                        </label>
                        <input 
                            type="text" 
                            name="marca" 
                            placeholder="Ej: Yamaha, Honda..."
                            class="w-full px-3 py-2 border border-gray-300 rounded-md dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                            value="${this.currentFilters.marca || ''}"
                        />
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                            Modelo
                        </label>
                        <input 
                            type="text" 
                            name="modelo" 
                            placeholder="Ej: MT-03, CB500F..."
                            class="w-full px-3 py-2 border border-gray-300 rounded-md dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                            value="${this.currentFilters.modelo || ''}"
                        />
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                            Estado
                        </label>
                        <select 
                            name="estado" 
                            class="w-full px-3 py-2 border border-gray-300 rounded-md dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                        >
                            <option value="">Todos los estados</option>
                            <option value="stock" ${this.currentFilters.estado === 'stock' ? 'selected' : ''}>En Stock</option>
                            <option value="reservada" ${this.currentFilters.estado === 'reservada' ? 'selected' : ''}>Reservada</option>
                            <option value="vendida" ${this.currentFilters.estado === 'vendida' ? 'selected' : ''}>Vendida</option>
                            <option value="mantenimiento" ${this.currentFilters.estado === 'mantenimiento' ? 'selected' : ''}>Mantenimiento</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                            Origen
                        </label>
                        <select 
                            name="origen" 
                            class="w-full px-3 py-2 border border-gray-300 rounded-md dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                        >
                            <option value="">Todos los orígenes</option>
                            <option value="propia" ${this.currentFilters.origen === 'propia' ? 'selected' : ''}>Propia</option>
                            <option value="consignacion" ${this.currentFilters.origen === 'consignacion' ? 'selected' : ''}>Consignación</option>
                            <option value="parte_pago" ${this.currentFilters.origen === 'parte_pago' ? 'selected' : ''}>Parte de Pago</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                            Cliente
                        </label>
                        <input 
                            type="text" 
                            name="cliente" 
                            placeholder="Nombre del cliente..."
                            class="w-full px-3 py-2 border border-gray-300 rounded-md dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                            value="${this.currentFilters.cliente || ''}"
                        />
                    </div>
                    <div class="flex gap-3 pt-4">
                        <button 
                            type="submit" 
                            class="flex-1 bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700 transition-colors"
                        >
                            Aplicar Filtros
                        </button>
                        <button 
                            type="button" 
                            class="flex-1 bg-gray-300 text-gray-700 px-4 py-2 rounded-md hover:bg-gray-400 transition-colors"
                            onclick="this.closest('.fixed').remove()"
                        >
                            Cancelar
                        </button>
                    </div>
                </form>
            </div>
        `;

        document.body.appendChild(modal);

        // Manejar envío del formulario
        modal.querySelector('#filter-form').addEventListener('submit', (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            const filters = {};
            
            for (let [key, value] of formData.entries()) {
                if (value.trim()) {
                    filters[key] = value.trim();
                }
            }

            this.currentFilters = filters;
            this.loadMotos(filters);
            modal.remove();
        });

        // Cerrar modal al hacer clic fuera
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                modal.remove();
            }
        });
    }

    clearFilters() {
        this.currentFilters = {};
        this.loadMotos();
    }

    updateStats() {
        // Actualizar estadísticas en la interfaz si existe un elemento para ello
        const statsContainer = document.querySelector('[data-stats="motos"]');
        if (statsContainer) {
            const enStock = this.motos.filter(m => m.estado === 'stock').length;
            const vendidas = this.motos.filter(m => m.estado === 'vendida').length;
            const reservadas = this.motos.filter(m => m.estado === 'reservada').length;
            const mantenimiento = this.motos.filter(m => m.estado === 'mantenimiento').length;

            statsContainer.innerHTML = `
                <div class="text-sm text-gray-600 dark:text-gray-400">
                    Total: ${this.motos.length} • 
                    En Stock: ${enStock} • 
                    Vendidas: ${vendidas} • 
                    Reservadas: ${reservadas}
                    ${mantenimiento > 0 ? ` • Mantenimiento: ${mantenimiento}` : ''}
                </div>
            `;
        }
    }

    showError(message) {
        // Mostrar mensaje de error
        const tbody = document.querySelector('table tbody');
        if (tbody) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="4" class="py-8 text-center">
                        <div class="text-red-500 dark:text-red-400">
                            <p class="text-lg mb-2">⚠️ Error</p>
                            <p class="text-sm">${message}</p>
                        </div>
                    </td>
                </tr>
            `;
        }
    }

    debounce(func, wait) {
        clearTimeout(this.debounceTimer);
        this.debounceTimer = setTimeout(func, wait);
    }

    filterBySearch(query) {
        if (!query.trim()) {
            this.clearFilters();
            return;
        }

        this.loadMotos({
            ...this.currentFilters,
            marca: query
        });
    }
}

// Inicializar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', () => {
    if (document.querySelector('table')) {
        window.tableMotos = new TableMotos();
    }
});

export default TableMotos;
