/**
 * GRÁFICOS EMPRESARIALES DINÁMICOS
 * Sistema de análisis de datos por empresa con gráficos interactivos
 * Especializado en análisis de clientes, regiones y toma de decisiones
 */

class GraficosEmpresa {
  constructor() {
    this.empresaId = localStorage.getItem('id_empresa');
    this.token = localStorage.getItem('authToken') || localStorage.getItem('token');
    this.baseURL = window.API_CONFIG?.BASE_URL || 'http://localhost:3001';
    this.charts = {};
    this.datosCache = {};
    
    console.log('📊 Inicializando Gráficos Empresariales para empresa:', this.empresaId);
    this.init();
  }

  async init() {
    if (!this.empresaId || !this.token) {
      console.warn('⚠️ Sin datos de empresa o token, usando datos de ejemplo');
      this.empresaId = '1'; // Empresa por defecto para demo
    }
    
    await this.cargarDatosEmpresa();
    this.configurarEventListeners();
  }

  // ==================== CARGA DE DATOS EMPRESARIALES ====================
  
  async cargarDatosEmpresa() {
    try {
      console.log('🔄 Cargando datos empresariales...');
      
      // Cargar datos en paralelo
      const [
        datosClientes,
        datosVentas,
        datosInventario,
        datosRegiones
      ] = await Promise.allSettled([
        this.obtenerDatosClientes(),
        this.obtenerDatosVentas(),
        this.obtenerDatosInventario(),
        this.obtenerDatosRegiones()
      ]);

      // Procesar resultados
      this.datosCache.clientes = datosClientes.status === 'fulfilled' ? datosClientes.value : this.obtenerDatosEjemploClientes();
      this.datosCache.ventas = datosVentas.status === 'fulfilled' ? datosVentas.value : this.obtenerDatosEjemploVentas();
      this.datosCache.inventario = datosInventario.status === 'fulfilled' ? datosInventario.value : this.obtenerDatosEjemploInventario();
      this.datosCache.regiones = datosRegiones.status === 'fulfilled' ? datosRegiones.value : this.obtenerDatosEjemploRegiones();

      console.log('✅ Datos empresariales cargados:', {
        clientes: this.datosCache.clientes.length,
        ventas: this.datosCache.ventas.length,
        inventario: this.datosCache.inventario.length,
        regiones: this.datosCache.regiones.length
      });

      // Generar gráficos automáticamente
      this.generarTodosLosGraficos();
      
    } catch (error) {
      console.error('❌ Error cargando datos empresariales:', error);
      this.usarDatosEjemplo();
    }
  }

  async obtenerDatosClientes() {
    const response = await fetch(`${this.baseURL}/api/clientes?empresa=${this.empresaId}`, {
      headers: { 'Authorization': `Bearer ${this.token}` }
    });
    
    if (response.ok) {
      const data = await response.json();
      return data.data || data;
    }
    throw new Error('Error obteniendo clientes');
  }

  async obtenerDatosVentas() {
    const response = await fetch(`${this.baseURL}/api/ventas?empresa=${this.empresaId}`, {
      headers: { 'Authorization': `Bearer ${this.token}` }
    });
    
    if (response.ok) {
      const data = await response.json();
      return data.data || data;
    }
    throw new Error('Error obteniendo ventas');
  }

  async obtenerDatosInventario() {
    const response = await fetch(`${this.baseURL}/api/motos?empresa=${this.empresaId}`, {
      headers: { 'Authorization': `Bearer ${this.token}` }
    });
    
    if (response.ok) {
      const data = await response.json();
      return data.data || data;
    }
    throw new Error('Error obteniendo inventario');
  }

  async obtenerDatosRegiones() {
    const response = await fetch(`${this.baseURL}/api/regiones`, {
      headers: { 'Authorization': `Bearer ${this.token}` }
    });
    
    if (response.ok) {
      const data = await response.json();
      return data.data || data;
    }
    throw new Error('Error obteniendo regiones');
  }

  // ==================== GRÁFICOS DE CLIENTES POR REGIÓN ====================
  
  crearGraficoClientesPorRegion(containerId = 'chartClientesRegion') {
    const container = document.getElementById(containerId);
    if (!container) {
      console.warn(`⚠️ Contenedor ${containerId} no encontrado`);
      return;
    }

    const datosRegiones = this.analizarClientesPorRegion();
    
    if (typeof Chart === 'undefined') {
      console.error('❌ Chart.js no está disponible');
      return;
    }

    // Destruir gráfico anterior si existe
    if (this.charts[containerId]) {
      this.charts[containerId].destroy();
    }

    const ctx = container.getContext('2d');
    
    this.charts[containerId] = new Chart(ctx, {
      type: 'doughnut',
      data: {
        labels: datosRegiones.labels,
        datasets: [{
          label: 'Clientes por Región',
          data: datosRegiones.valores,
          backgroundColor: [
            '#3B82F6', // Azul
            '#10B981', // Verde
            '#F59E0B', // Amarillo
            '#EF4444', // Rojo
            '#8B5CF6', // Púrpura
            '#F97316', // Naranja
            '#06B6D4', // Cian
            '#84CC16', // Lima
            '#EC4899', // Rosa
            '#6B7280'  // Gris
          ].slice(0, datosRegiones.labels.length),
          borderWidth: 2,
          borderColor: '#ffffff'
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: {
            position: 'right',
            labels: {
              usePointStyle: true,
              padding: 20,
              generateLabels: function(chart) {
                const data = chart.data;
                if (data.labels.length && data.datasets.length) {
                  return data.labels.map((label, i) => {
                    const meta = chart.getDatasetMeta(0);
                    const style = meta.controller.getStyle(i);
                    const value = data.datasets[0].data[i];
                    const total = data.datasets[0].data.reduce((a, b) => a + b, 0);
                    const percentage = ((value / total) * 100).toFixed(1);
                    
                    return {
                      text: `${label}: ${value} (${percentage}%)`,
                      fillStyle: style.backgroundColor,
                      strokeStyle: style.borderColor,
                      lineWidth: style.borderWidth,
                      pointStyle: 'circle',
                      hidden: false,
                      index: i
                    };
                  });
                }
                return [];
              }
            }
          },
          tooltip: {
            callbacks: {
              label: function(context) {
                const label = context.label || '';
                const value = context.parsed || 0;
                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                const percentage = ((value / total) * 100).toFixed(1);
                return `${label}: ${value} clientes (${percentage}%)`;
              }
            }
          },
          title: {
            display: true,
            text: 'Distribución de Clientes por Región',
            font: {
              size: 16,
              weight: 'bold'
            }
          }
        },
        animation: {
          animateScale: true,
          animateRotate: true
        }
      }
    });

    // Agregar información adicional
    this.mostrarInsightsRegiones(datosRegiones, containerId);
  }

  analizarClientesPorRegion() {
    const clientes = this.datosCache.clientes || [];
    const regiones = this.datosCache.regiones || [];
    
    // Crear mapa de regiones para nombres
    const mapaRegiones = {};
    regiones.forEach(region => {
      mapaRegiones[region.id] = region.nombre;
    });
    
    // Contar clientes por región
    const conteoRegiones = {};
    clientes.forEach(cliente => {
      const regionId = cliente.id_region;
      const regionNombre = mapaRegiones[regionId] || `Región ${regionId}` || 'Sin Región';
      
      conteoRegiones[regionNombre] = (conteoRegiones[regionNombre] || 0) + 1;
    });
    
    // Ordenar por cantidad (mayor a menor)
    const regionesOrdenadas = Object.entries(conteoRegiones)
      .sort(([,a], [,b]) => b - a);
    
    return {
      labels: regionesOrdenadas.map(([nombre]) => nombre),
      valores: regionesOrdenadas.map(([, cantidad]) => cantidad),
      total: clientes.length,
      detalle: regionesOrdenadas
    };
  }

  mostrarInsightsRegiones(datos, containerId) {
    const insightsContainer = document.getElementById(`insights-${containerId}`);
    if (!insightsContainer) return;

    const regionPrincipal = datos.detalle[0];
    const porcentajePrincipal = ((regionPrincipal[1] / datos.total) * 100).toFixed(1);
    
    const regionesConPocosClientes = datos.detalle.filter(([, cantidad]) => cantidad <= 2);
    
    const insights = [
      `🏆 ${regionPrincipal[0]} concentra el ${porcentajePrincipal}% de tus clientes (${regionPrincipal[1]} clientes)`,
      `📍 Tienes presencia en ${datos.labels.length} regiones diferentes`,
      `⚡ ${regionesConPocosClientes.length} regiones tienen potencial de crecimiento (≤2 clientes)`,
      `🎯 Oportunidad: Enfocar marketing en regiones con pocos clientes`
    ];

    insightsContainer.innerHTML = `
      <div class="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4 mt-4">
        <h4 class="font-semibold text-blue-800 dark:text-blue-200 mb-3">
          <i class="fas fa-chart-pie mr-2"></i>Análisis de Distribución Regional
        </h4>
        <ul class="space-y-2 text-sm text-blue-700 dark:text-blue-300">
          ${insights.map(insight => `<li><i class="fas fa-arrow-right mr-2"></i>${insight}</li>`).join('')}
        </ul>
      </div>
    `;
  }

  // ==================== GRÁFICO DE VENTAS POR REGIÓN ====================
  
  crearGraficoVentasPorRegion(containerId = 'chartVentasRegion') {
    const container = document.getElementById(containerId);
    if (!container || typeof Chart === 'undefined') return;

    const datosVentasRegion = this.analizarVentasPorRegion();
    
    if (this.charts[containerId]) {
      this.charts[containerId].destroy();
    }

    const ctx = container.getContext('2d');
    
    this.charts[containerId] = new Chart(ctx, {
      type: 'bar',
      data: {
        labels: datosVentasRegion.labels,
        datasets: [
          {
            label: 'Cantidad de Ventas',
            data: datosVentasRegion.cantidades,
            backgroundColor: 'rgba(59, 130, 246, 0.8)',
            borderColor: 'rgb(59, 130, 246)',
            borderWidth: 1,
            yAxisID: 'y'
          },
          {
            label: 'Monto Total (M$)',
            data: datosVentasRegion.montos.map(monto => monto / 1000000),
            backgroundColor: 'rgba(16, 185, 129, 0.8)',
            borderColor: 'rgb(16, 185, 129)',
            borderWidth: 1,
            yAxisID: 'y1'
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        interaction: {
          mode: 'index',
          intersect: false,
        },
        plugins: {
          title: {
            display: true,
            text: 'Ventas por Región - Cantidad vs Monto',
            font: { size: 16, weight: 'bold' }
          },
          legend: {
            position: 'top'
          },
          tooltip: {
            callbacks: {
              afterBody: function(tooltipItems) {
                const index = tooltipItems[0].dataIndex;
                const ticketPromedio = datosVentasRegion.montos[index] / datosVentasRegion.cantidades[index];
                return [`Ticket Promedio: $${ticketPromedio.toLocaleString('es-CL')}`];
              }
            }
          }
        },
        scales: {
          x: {
            display: true,
            title: {
              display: true,
              text: 'Regiones'
            }
          },
          y: {
            type: 'linear',
            display: true,
            position: 'left',
            title: {
              display: true,
              text: 'Cantidad de Ventas'
            }
          },
          y1: {
            type: 'linear',
            display: true,
            position: 'right',
            title: {
              display: true,
              text: 'Monto (Millones $)'
            },
            grid: {
              drawOnChartArea: false,
            },
          }
        }
      }
    });

    this.mostrarInsightsVentasRegion(datosVentasRegion, containerId);
  }

  analizarVentasPorRegion() {
    const ventas = this.datosCache.ventas || [];
    const clientes = this.datosCache.clientes || [];
    const regiones = this.datosCache.regiones || [];
    
    // Crear mapas de referencia
    const mapaClientes = {};
    clientes.forEach(cliente => {
      mapaClientes[cliente.id] = cliente;
    });
    
    const mapaRegiones = {};
    regiones.forEach(region => {
      mapaRegiones[region.id] = region.nombre;
    });
    
    // Analizar ventas por región
    const ventasPorRegion = {};
    
    ventas.forEach(venta => {
      const cliente = mapaClientes[venta.id_cliente];
      if (!cliente) return;
      
      const regionNombre = mapaRegiones[cliente.id_region] || 'Sin Región';
      
      if (!ventasPorRegion[regionNombre]) {
        ventasPorRegion[regionNombre] = {
          cantidad: 0,
          montoTotal: 0,
          ventas: []
        };
      }
      
      ventasPorRegion[regionNombre].cantidad++;
      ventasPorRegion[regionNombre].montoTotal += parseFloat(venta.precio_venta || 0);
      ventasPorRegion[regionNombre].ventas.push(venta);
    });
    
    // Ordenar por monto total
    const regionesOrdenadas = Object.entries(ventasPorRegion)
      .sort(([,a], [,b]) => b.montoTotal - a.montoTotal);
    
    return {
      labels: regionesOrdenadas.map(([nombre]) => nombre),
      cantidades: regionesOrdenadas.map(([, datos]) => datos.cantidad),
      montos: regionesOrdenadas.map(([, datos]) => datos.montoTotal),
      detalle: regionesOrdenadas
    };
  }

  mostrarInsightsVentasRegion(datos, containerId) {
    const insightsContainer = document.getElementById(`insights-${containerId}`);
    if (!insightsContainer) return;

    const regionTop = datos.detalle[0];
    const montoTotal = datos.montos.reduce((a, b) => a + b, 0);
    const ventasTotal = datos.cantidades.reduce((a, b) => a + b, 0);
    const ticketPromedio = montoTotal / ventasTotal;
    
    const insights = [
      `💰 ${regionTop[0]} genera el ${((regionTop[1].montoTotal / montoTotal) * 100).toFixed(1)}% de tus ingresos`,
      `📈 Ticket promedio general: $${ticketPromedio.toLocaleString('es-CL')}`,
      `🌟 ${datos.labels.length} regiones con ventas activas`,
      `🎯 Enfoca recursos en las 3 regiones top para maximizar ROI`
    ];

    insightsContainer.innerHTML = `
      <div class="bg-green-50 dark:bg-green-900/20 rounded-lg p-4 mt-4">
        <h4 class="font-semibold text-green-800 dark:text-green-200 mb-3">
          <i class="fas fa-chart-bar mr-2"></i>Análisis de Ventas Regionales
        </h4>
        <ul class="space-y-2 text-sm text-green-700 dark:text-green-300">
          ${insights.map(insight => `<li><i class="fas fa-arrow-right mr-2"></i>${insight}</li>`).join('')}
        </ul>
      </div>
    `;
  }

  // ==================== GRÁFICO DE INVENTARIO POR REGIÓN ====================
  
  crearGraficoInventarioPorRegion(containerId = 'chartInventarioRegion') {
    const container = document.getElementById(containerId);
    if (!container || typeof Chart === 'undefined') return;

    const datosInventario = this.analizarInventarioPorOrigen();
    
    if (this.charts[containerId]) {
      this.charts[containerId].destroy();
    }

    const ctx = container.getContext('2d');
    
    this.charts[containerId] = new Chart(ctx, {
      type: 'polarArea',
      data: {
        labels: datosInventario.labels,
        datasets: [{
          label: 'Motos por Origen',
          data: datosInventario.valores,
          backgroundColor: [
            'rgba(59, 130, 246, 0.7)',   // Azul
            'rgba(16, 185, 129, 0.7)',   // Verde  
            'rgba(245, 158, 11, 0.7)',   // Amarillo
            'rgba(239, 68, 68, 0.7)',    // Rojo
            'rgba(139, 92, 246, 0.7)',   // Púrpura
            'rgba(6, 182, 212, 0.7)'     // Cian
          ],
          borderColor: [
            'rgb(59, 130, 246)',
            'rgb(16, 185, 129)', 
            'rgb(245, 158, 11)',
            'rgb(239, 68, 68)',
            'rgb(139, 92, 246)',
            'rgb(6, 182, 212)'
          ],
          borderWidth: 2
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          title: {
            display: true,
            text: 'Distribución de Inventario por Origen',
            font: { size: 16, weight: 'bold' }
          },
          legend: {
            position: 'bottom'
          },
          tooltip: {
            callbacks: {
              label: function(context) {
                const label = context.label || '';
                const value = context.parsed || 0;
                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                const percentage = ((value / total) * 100).toFixed(1);
                return `${label}: ${value} motos (${percentage}%)`;
              }
            }
          }
        },
        scales: {
          r: {
            beginAtZero: true,
            ticks: {
              stepSize: 1
            }
          }
        },
        animation: {
          animateScale: true,
          animateRotate: true
        }
      }
    });

    this.mostrarInsightsInventario(datosInventario, containerId);
  }

  analizarInventarioPorOrigen() {
    const inventario = this.datosCache.inventario || [];
    
    const origenesCount = {};
    const origenes = ['PROPIA', 'CONSIGNACION', 'PARTE_PAGO', 'COMPRA', 'IMPORTACION'];
    
    // Inicializar contadores
    origenes.forEach(origen => {
      origenesCount[origen] = 0;
    });
    
    // Contar por origen
    inventario.forEach(moto => {
      const origen = moto.estado_comercial?.origen || 'PROPIA';
      if (origenesCount.hasOwnProperty(origen)) {
        origenesCount[origen]++;
      } else {
        origenesCount['PROPIA']++; // Default
      }
    });
    
    // Filtrar solo los que tienen valores > 0
    const origenesFiltrados = Object.entries(origenesCount)
      .filter(([, cantidad]) => cantidad > 0)
      .sort(([,a], [,b]) => b - a);
    
    return {
      labels: origenesFiltrados.map(([origen]) => this.formatearOrigen(origen)),
      valores: origenesFiltrados.map(([, cantidad]) => cantidad),
      detalle: origenesFiltrados
    };
  }

  formatearOrigen(origen) {
    const formatos = {
      'PROPIA': 'Motos Propias',
      'CONSIGNACION': 'Consignación',
      'PARTE_PAGO': 'Parte de Pago',
      'COMPRA': 'Compra Directa',
      'IMPORTACION': 'Importación'
    };
    return formatos[origen] || origen;
  }

  mostrarInsightsInventario(datos, containerId) {
    const insightsContainer = document.getElementById(`insights-${containerId}`);
    if (!insightsContainer) return;

    const totalMotos = datos.valores.reduce((a, b) => a + b, 0);
    const origenPrincipal = datos.detalle[0];
    
    const insights = [
      `🏍️ Total de ${totalMotos} motos en inventario`,
      `📊 ${origenPrincipal[0]}: ${origenPrincipal[1]} motos (${((origenPrincipal[1] / totalMotos) * 100).toFixed(1)}%)`,
      `🔄 Diversificación en ${datos.labels.length} tipos de origen`,
      `💡 Considera balancear entre consignación y compra directa`
    ];

    insightsContainer.innerHTML = `
      <div class="bg-purple-50 dark:bg-purple-900/20 rounded-lg p-4 mt-4">
        <h4 class="font-semibold text-purple-800 dark:text-purple-200 mb-3">
          <i class="fas fa-warehouse mr-2"></i>Análisis de Inventario
        </h4>
        <ul class="space-y-2 text-sm text-purple-700 dark:text-purple-300">
          ${insights.map(insight => `<li><i class="fas fa-arrow-right mr-2"></i>${insight}</li>`).join('')}
        </ul>
      </div>
    `;
  }

  // ==================== GENERACIÓN AUTOMÁTICA DE GRÁFICOS ====================
  
  generarTodosLosGraficos() {
    console.log('🎨 Generando todos los gráficos empresariales...');
    
    // Verificar que Chart.js esté disponible
    if (typeof Chart === 'undefined') {
      console.error('❌ Chart.js no está disponible. Cargando...');
      this.cargarChartJS().then(() => {
        this.generarGraficos();
      });
      return;
    }
    
    this.generarGraficos();
  }

  generarGraficos() {
    try {
      // Gráficos principales
      this.crearGraficoClientesPorRegion();
      this.crearGraficoVentasPorRegion(); 
      this.crearGraficoInventarioPorRegion();
      
      // Generar gráficos adicionales si existen contenedores
      this.generarGraficosAdicionales();
      
      console.log('✅ Todos los gráficos generados exitosamente');
      
    } catch (error) {
      console.error('❌ Error generando gráficos:', error);
    }
  }

  generarGraficosAdicionales() {
    // Gráfico de evolución temporal de clientes
    if (document.getElementById('chartEvolucionClientes')) {
      this.crearGraficoEvolucionClientes();
    }
    
    // Gráfico comparativo de marcas más vendidas
    if (document.getElementById('chartMarcasVendidas')) {
      this.crearGraficoMarcasVendidas();
    }
    
    // Gráfico de análisis de precios
    if (document.getElementById('chartAnalisisPrecios')) {
      this.crearGraficoAnalisisPrecios();
    }
  }

  // ==================== CONFIGURACIÓN Y EVENTOS ====================
  
  configurarEventListeners() {
    // Botón de actualización de datos
    const btnActualizar = document.getElementById('btnActualizarGraficos');
    if (btnActualizar) {
      btnActualizar.addEventListener('click', () => {
        this.actualizarTodosLosDatos();
      });
    }
    
    // Selector de filtros temporales
    const selectorPeriodo = document.getElementById('selectorPeriodo');
    if (selectorPeriodo) {
      selectorPeriodo.addEventListener('change', (e) => {
        this.filtrarPorPeriodo(e.target.value);
      });
    }
    
    // Botón de exportar datos
    const btnExportar = document.getElementById('btnExportarDatos');
    if (btnExportar) {
      btnExportar.addEventListener('click', () => {
        this.exportarDatosEmpresa();
      });
    }
  }

  async actualizarTodosLosDatos() {
    console.log('🔄 Actualizando todos los datos...');
    
    // Limpiar cache
    this.datosCache = {};
    
    // Mostrar indicador de carga
    this.mostrarIndicadorCarga(true);
    
    try {
      await this.cargarDatosEmpresa();
      
      if (typeof Swal !== 'undefined') {
        Swal.fire({
          title: 'Datos Actualizados',
          text: 'Los gráficos han sido actualizados con los últimos datos',
          icon: 'success',
          timer: 2000,
          showConfirmButton: false
        });
      }
      
    } catch (error) {
      console.error('❌ Error actualizando datos:', error);
      
      if (typeof Swal !== 'undefined') {
        Swal.fire({
          title: 'Error',
          text: 'No se pudieron actualizar los datos. Usando datos locales.',
          icon: 'warning',
          timer: 3000,
          showConfirmButton: false
        });
      }
    } finally {
      this.mostrarIndicadorCarga(false);
    }
  }

  mostrarIndicadorCarga(mostrar) {
    const indicador = document.getElementById('indicadorCargaGraficos');
    if (indicador) {
      indicador.style.display = mostrar ? 'block' : 'none';
    }
  }

  // ==================== DATOS DE EJEMPLO ====================
  
  usarDatosEjemplo() {
    console.log('📊 Usando datos de ejemplo para los gráficos...');
    
    this.datosCache = {
      clientes: this.obtenerDatosEjemploClientes(),
      ventas: this.obtenerDatosEjemploVentas(),
      inventario: this.obtenerDatosEjemploInventario(),
      regiones: this.obtenerDatosEjemploRegiones()
    };
    
    this.generarTodosLosGraficos();
  }

  obtenerDatosEjemploClientes() {
    return [
      { id: 1, nombres: 'Juan', apellidos: 'Pérez', id_region: 1, tipo_cliente: 'NATURAL' },
      { id: 2, nombres: 'María', apellidos: 'González', id_region: 1, tipo_cliente: 'NATURAL' },
      { id: 3, nombres: 'Carlos', apellidos: 'Silva', id_region: 2, tipo_cliente: 'NATURAL' },
      { id: 4, nombres: 'Ana', apellidos: 'Martínez', id_region: 3, tipo_cliente: 'NATURAL' },
      { id: 5, nombres: 'Luis', apellidos: 'Rodríguez', id_region: 1, tipo_cliente: 'NATURAL' },
      { id: 6, nombres: 'Sofia', apellidos: 'López', id_region: 2, tipo_cliente: 'NATURAL' },
      { id: 7, nombres: 'Miguel', apellidos: 'Torres', id_region: 4, tipo_cliente: 'EMPRESA' },
      { id: 8, nombres: 'Carmen', apellidos: 'Ruiz', id_region: 3, tipo_cliente: 'NATURAL' },
      { id: 9, nombres: 'Pedro', apellidos: 'Morales', id_region: 1, tipo_cliente: 'NATURAL' },
      { id: 10, nombres: 'Laura', apellidos: 'Herrera', id_region: 5, tipo_cliente: 'NATURAL' }
    ];
  }

  obtenerDatosEjemploVentas() {
    return [
      { id: 1, id_cliente: 1, precio_venta: 8500000, fecha_venta: '2024-01-15' },
      { id: 2, id_cliente: 3, precio_venta: 12000000, fecha_venta: '2024-02-10' },
      { id: 3, id_cliente: 2, precio_venta: 6500000, fecha_venta: '2024-02-20' },
      { id: 4, id_cliente: 5, precio_venta: 15000000, fecha_venta: '2024-03-05' },
      { id: 5, id_cliente: 7, precio_venta: 22000000, fecha_venta: '2024-03-15' },
      { id: 6, id_cliente: 4, precio_venta: 9500000, fecha_venta: '2024-04-02' },
      { id: 7, id_cliente: 6, precio_venta: 11000000, fecha_venta: '2024-04-18' },
      { id: 8, id_cliente: 8, precio_venta: 7800000, fecha_venta: '2024-05-10' }
    ];
  }

  obtenerDatosEjemploInventario() {
    return [
      { id_moto: 1, estado_comercial: { origen: 'PROPIA' } },
      { id_moto: 2, estado_comercial: { origen: 'CONSIGNACION' } },
      { id_moto: 3, estado_comercial: { origen: 'PROPIA' } },
      { id_moto: 4, estado_comercial: { origen: 'PARTE_PAGO' } },
      { id_moto: 5, estado_comercial: { origen: 'CONSIGNACION' } },
      { id_moto: 6, estado_comercial: { origen: 'PROPIA' } },
      { id_moto: 7, estado_comercial: { origen: 'COMPRA' } },
      { id_moto: 8, estado_comercial: { origen: 'PROPIA' } }
    ];
  }

  obtenerDatosEjemploRegiones() {
    return [
      { id: 1, nombre: 'Metropolitana' },
      { id: 2, nombre: 'Valparaíso' },
      { id: 3, nombre: 'Biobío' },
      { id: 4, nombre: 'La Araucanía' },
      { id: 5, nombre: 'Los Lagos' }
    ];
  }

  // ==================== UTILIDADES ====================
  
  async cargarChartJS() {
    return new Promise((resolve, reject) => {
      if (typeof Chart !== 'undefined') {
        resolve();
        return;
      }
      
      const script = document.createElement('script');
      script.src = 'https://cdn.jsdelivr.net/npm/chart.js';
      script.onload = resolve;
      script.onerror = reject;
      document.head.appendChild(script);
    });
  }

  exportarDatosEmpresa() {
    const datosExport = {
      empresa: this.empresaId,
      fecha: new Date().toISOString(),
      clientes: this.datosCache.clientes,
      ventas: this.datosCache.ventas,
      inventario: this.datosCache.inventario,
      analisis: {
        clientesPorRegion: this.analizarClientesPorRegion(),
        ventasPorRegion: this.analizarVentasPorRegion(),
        inventarioPorOrigen: this.analizarInventarioPorOrigen()
      }
    };
    
    const dataStr = JSON.stringify(datosExport, null, 2);
    const dataBlob = new Blob([dataStr], {type: 'application/json'});
    
    const url = URL.createObjectURL(dataBlob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `analisis-empresa-${this.empresaId}-${new Date().toISOString().split('T')[0]}.json`;
    link.click();
    
    URL.revokeObjectURL(url);
  }

  destruir() {
    // Limpiar todos los gráficos
    Object.values(this.charts).forEach(chart => {
      if (chart && typeof chart.destroy === 'function') {
        chart.destroy();
      }
    });
    
    this.charts = {};
    this.datosCache = {};
    
    console.log('🧹 Gráficos empresariales destruidos');
  }
}

// ==================== INICIALIZACIÓN GLOBAL ====================

// Instancia global para uso en el frontend
window.GraficosEmpresa = GraficosEmpresa;

// Auto-inicialización cuando el DOM está listo
document.addEventListener('DOMContentLoaded', function() {
  // Solo inicializar en páginas que tengan gráficos empresariales
  if (document.getElementById('chartClientesRegion') || 
      document.getElementById('chartVentasRegion') || 
      document.getElementById('chartInventarioRegion')) {
    
    console.log('📊 Inicializando sistema de gráficos empresariales...');
    window.graficosEmpresaInstance = new GraficosEmpresa();
  }
});

// Exportar para uso como módulo
if (typeof module !== 'undefined' && module.exports) {
  module.exports = GraficosEmpresa;
}
