/**
 * OPERACIONES GLOBAL - Funciones compartidas entre todos los módulos de operaciones
 * ====================================================================
 */

// Variables globales
let operacionActual = null;
let clienteSeleccionado = null;
let motoSeleccionada = null;
let detallesOperacion = {};
let trabajadorActual = null;

// ====================================================================
// INICIALIZACIÓN Y AUTENTICACIÓN
// ====================================================================

/**
 * Verificar autenticación del usuario
 */
function verificarAutenticacion() {
  if (!localStorage.getItem('id_usuario')) {
    Swal.fire({
      icon: 'warning',
      title: 'Acceso restringido',
      text: 'Debes iniciar sesión para acceder a esta página.',
      allowOutsideClick: false,
      allowEscapeKey: false,
      showConfirmButton: true,
      confirmButtonText: 'Ir al login'
    }).then(() => {
      window.location.href = '../signin.html';
    });
    return false;
  }
  return true;
}

/**
 * Obtener trabajador actual desde localStorage o token
 */
function obtenerTrabajadorActual() {
  const id_usuario = localStorage.getItem('id_usuario');
  const nombre_usuario = localStorage.getItem('nombre_usuario');
  const email_usuario = localStorage.getItem('email_usuario');
  const rol_usuario = localStorage.getItem('rol_usuario');

  if (id_usuario) {
    trabajadorActual = {
      id: id_usuario,
      nombre: nombre_usuario || 'Usuario',
      email: email_usuario || '',
      rol: rol_usuario || 'trabajador'
    };
    console.log('Trabajador actual:', trabajadorActual);
  } else {
    console.error('No se encontró información del usuario en localStorage');
    const token = localStorage.getItem('token') || localStorage.getItem('authToken');
    if (token) {
      try {
        const payload = JSON.parse(atob(token.split('.')[1]));
        trabajadorActual = {
          id: payload.id || payload.id_usuario,
          nombre: payload.nombre || payload.nombre_usuario,
          email: payload.email || payload.email_usuario,
          rol: payload.rol || payload.rol_usuario || 'trabajador'
        };
        console.log('Trabajador obtenido desde token:', trabajadorActual);
      } catch (e) {
        console.error('Error decodificando token:', e);
      }
    }
  }
  return trabajadorActual;
}

// ====================================================================
// CLIENTES - Funciones para gestión de clientes
// ====================================================================

/**
 * Buscar clientes en tiempo real
 */
async function buscarClientesEnTiempoReal(termino) {
  const resultadosDiv = document.getElementById('resultadosClientes');
  
  if (!termino || termino.length < 2) {
    resultadosDiv.classList.add('hidden');
    return;
  }

  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const id_empresa = localStorage.getItem('id_empresa');
    
    const response = await fetch(`http://localhost:3001/api/clientes/buscar?termino=${encodeURIComponent(termino)}&id_empresa=${id_empresa}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (response.ok) {
      const clientes = await response.json();
      mostrarResultadosClientes(clientes);
    } else {
      console.error('Error en la búsqueda de clientes');
      resultadosDiv.classList.add('hidden');
    }
  } catch (error) {
    console.error('Error buscando clientes:', error);
    resultadosDiv.classList.add('hidden');
  }
}

/**
 * Mostrar resultados de búsqueda de clientes
 */
function mostrarResultadosClientes(clientes) {
  const resultadosDiv = document.getElementById('resultadosClientes');
  
  if (clientes.length === 0) {
    resultadosDiv.innerHTML = '<p class="p-3 text-sm text-gray-500 dark:text-gray-400">No se encontraron clientes</p>';
    resultadosDiv.classList.remove('hidden');
    return;
  }

  const html = clientes.map(cliente => `
    <div class="p-3 hover:bg-gray-50 dark:hover:bg-gray-600 cursor-pointer border-b border-gray-200 dark:border-gray-600 last:border-b-0" 
         onclick="seleccionarCliente(${JSON.stringify(cliente).replace(/"/g, '&quot;')})">
      <div class="flex justify-between items-start">
        <div>
          <p class="font-medium text-sm text-gray-900 dark:text-white">${cliente.nombres} ${cliente.apellidos}</p>
          <p class="text-xs text-gray-600 dark:text-gray-400">RUT: ${cliente.cedula}</p>
          ${cliente.telefono ? `<p class="text-xs text-gray-600 dark:text-gray-400">📞 ${cliente.telefono}</p>` : ''}
        </div>
        <span class="text-xs bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-300 px-2 py-1 rounded">
          ${cliente.tipo_cliente}
        </span>
      </div>
    </div>
  `).join('');

  resultadosDiv.innerHTML = html;
  resultadosDiv.classList.remove('hidden');
}

/**
 * Seleccionar un cliente de los resultados de búsqueda
 */
function seleccionarCliente(cliente) {
  clienteSeleccionado = cliente;
  
  document.getElementById('resultadosClientes').classList.add('hidden');
  mostrarClienteSeleccionado(cliente);
  document.getElementById('buscarCliente').value = `${cliente.nombres} ${cliente.apellidos} - ${cliente.cedula}`;
  
  console.log('Cliente seleccionado:', cliente);
}

/**
 * Mostrar información del cliente seleccionado
 */
function mostrarClienteSeleccionado(cliente) {
  let infoDiv = document.getElementById('infoClienteSeleccionado');
  if (!infoDiv) {
    infoDiv = document.createElement('div');
    infoDiv.id = 'infoClienteSeleccionado';
    document.getElementById('buscarCliente').parentNode.insertAdjacentElement('afterend', infoDiv);
  }

  infoDiv.innerHTML = `
    <div class="mt-3 p-3 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-700 rounded-lg">
      <div class="flex items-center justify-between">
        <div>
          <p class="font-medium text-green-800 dark:text-green-300">${cliente.nombres} ${cliente.apellidos}</p>
          <p class="text-sm text-green-700 dark:text-green-400">RUT: ${cliente.cedula} | Tipo: ${cliente.tipo_cliente}</p>
          ${cliente.telefono ? `<p class="text-sm text-green-700 dark:text-green-400">📞 ${cliente.telefono}</p>` : ''}
        </div>
        <button onclick="limpiarSeleccionCliente()" class="text-green-600 dark:text-green-400 hover:text-green-800 dark:hover:text-green-200">
          <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
          </svg>
        </button>
      </div>
    </div>
  `;
}

/**
 * Limpiar selección de cliente
 */
function limpiarSeleccionCliente() {
  clienteSeleccionado = null;
  document.getElementById('buscarCliente').value = '';
  const infoDiv = document.getElementById('infoClienteSeleccionado');
  if (infoDiv) {
    infoDiv.remove();
  }
  document.getElementById('resultadosClientes').classList.add('hidden');
}

// ====================================================================
// VALIDACIONES - Funciones de validación para formularios
// ====================================================================

/**
 * Formatear RUT chileno
 */
function formatearRutChileno(input) {
  let rut = input.value.replace(/[^0-9kK]/g, '');
  
  if (rut.length > 1) {
    rut = rut.slice(0, -1) + '-' + rut.slice(-1);
  }
  
  if (rut.length > 4) {
    rut = rut.slice(0, -4) + '.' + rut.slice(-4);
  }
  
  if (rut.length > 8) {
    rut = rut.slice(0, -8) + '.' + rut.slice(-8);
  }
  
  input.value = rut.toUpperCase();
}

/**
 * Validar RUT chileno
 */
function validarRutChileno(input) {
  const rut = input.value.replace(/[^0-9kK]/g, '');
  const errorDiv = document.getElementById('errorRut');
  
  if (!rut) {
    errorDiv.classList.add('hidden');
    return true;
  }
  
  if (rut.length < 2) {
    errorDiv.textContent = 'RUT debe tener al menos 2 caracteres';
    errorDiv.classList.remove('hidden');
    return false;
  }
  
  const cuerpo = rut.slice(0, -1);
  const dv = rut.slice(-1).toUpperCase();
  
  let suma = 0;
  let multiplo = 2;
  
  for (let i = cuerpo.length - 1; i >= 0; i--) {
    suma += parseInt(cuerpo.charAt(i)) * multiplo;
    multiplo = multiplo === 7 ? 2 : multiplo + 1;
  }
  
  const dvCalculado = 11 - (suma % 11);
  const dvFinal = dvCalculado === 11 ? '0' : dvCalculado === 10 ? 'K' : dvCalculado.toString();
  
  if (dv !== dvFinal) {
    errorDiv.textContent = 'RUT inválido';
    errorDiv.classList.remove('hidden');
    return false;
  }
  
  errorDiv.classList.add('hidden');
  return true;
}

/**
 * Formatear teléfono chileno
 */
function formatearTelefonoChileno(input) {
  let telefono = input.value.replace(/[^0-9]/g, '');
  
  if (telefono.startsWith('569')) {
    telefono = '+569 ' + telefono.slice(3, 7) + ' ' + telefono.slice(7, 11);
  } else if (telefono.startsWith('9') && telefono.length >= 8) {
    telefono = '+569 ' + telefono.slice(1, 5) + ' ' + telefono.slice(5, 9);
  }
  
  input.value = telefono;
}

// ====================================================================
// UTILIDADES - Funciones de utilidad general
// ====================================================================

/**
 * Validar el formato de una patente chilena
 */
function validarPatenteChile(patente) {
  if (!patente || patente.length < 6) {
    return { valida: false, formato: 'incompleto' };
  }

  // Patrones originales
  const formatoAntiguo = /^[A-Z]{2,4}[0-9]{2,3}$/;
  const formatoNuevo = /^[A-Z]{2,4}[0-9]{2,4}$/;
  
  // Patrones con guión (ABC123-4, ABCD12-3, etc.)
  const formatoAntiguoConGuion = /^[A-Z]{2,4}[0-9]{2,3}-[0-9]$/;
  const formatoNuevoConGuion = /^[A-Z]{2,4}[0-9]{2,4}-[0-9]$/;

  if (formatoAntiguo.test(patente)) {
    return { valida: true, formato: 'antiguo' };
  } else if (formatoNuevo.test(patente)) {
    return { valida: true, formato: 'nuevo' };
  } else if (formatoAntiguoConGuion.test(patente)) {
    return { valida: true, formato: 'antiguo con guión' };
  } else if (formatoNuevoConGuion.test(patente)) {
    return { valida: true, formato: 'nuevo con guión' };
  } else {
    return { valida: false, formato: 'inválido' };
  }
}

/**
 * Validar patente en tiempo real con feedback visual
 */
async function validarPatenteEnTiempoReal(patente, campoId = 'patenteMoto', feedbackId = 'patente-feedback') {
  const resultado = validarPatenteChile(patente);
  const campo = document.getElementById(campoId);
  const feedback = document.getElementById(feedbackId);

  if (resultado && resultado.valida) {
    if (campo) {
      campo.classList.remove('border-red-500');
      campo.classList.add('border-green-500');
    }
    if (feedback) {
      feedback.textContent = `✓ Patente ${resultado.formato} válida`;
      feedback.className = 'text-sm text-green-600';
    }
  } else {
    if (campo) {
      campo.classList.remove('border-green-500');
      campo.classList.add('border-red-500');
    }
    if (feedback) {
      feedback.textContent = '✗ Formato de patente inválido';
      feedback.className = 'text-sm text-red-600';
    }
  }

  return resultado;
}

/**
 * Validar que una patente sea única en la empresa
 */
async function validarPatenteUnica(patente, empresaId, apiRequestFn) {
  if (!patente) return true;
  
  try {
    const response = await apiRequestFn(`/motos/verificar-patente/${patente.toUpperCase()}?empresa=${empresaId}`);
    
    if (response.existe) {
      mostrarError('Patente Duplicada', 'Ya existe una moto con esta patente en tu empresa.');
      return false;
    }
    
    return true;
  } catch (error) {
    console.error('Error validando patente única:', error);
    return true; // Permitir continuar si hay error en la validación
  }
}

/**
 * Validar patente con API externa (opcional)
 */
async function validarPatenteExterna(patente) {
  try {
    // Aquí podrías integrar con API del Registro Civil u otros servicios
    // Por ahora solo validamos formato chileno
    return validarPatenteChile(patente);
  } catch (error) {
    console.warn('No se pudo validar patente externamente:', error);
    return { valida: true, formato: 'desconocido' };
  }
}

/**
 * Funciones de formateo reutilizables
 */
function formatearPrecio(precio) {
  if (!precio || precio === 0) return 'No especificado';
  return new Intl.NumberFormat('es-CL', {
    style: 'currency',
    currency: 'CLP',
    minimumFractionDigits: 0
  }).format(precio);
}

function formatearNumero(numero) {
  if (numero === null || numero === undefined) return '0';
  return new Intl.NumberFormat('es-CL').format(numero);
}

function formatearFecha(fecha) {
  if (!fecha) return 'No especificada';
  return new Date(fecha).toLocaleDateString('es-CL');
}

/**
 * Mostrar loading/spinner
 */
function mostrarLoading(mensaje = 'Cargando...') {
  Swal.fire({
    title: mensaje,
    allowOutsideClick: false,
    allowEscapeKey: false,
    showConfirmButton: false,
    didOpen: () => {
      Swal.showLoading();
    }
  });
}

/**
 * Cerrar loading
 */
function cerrarLoading() {
  Swal.close();
}

/**
 * Mostrar alerta de éxito
 */
function mostrarExito(titulo, mensaje = '') {
  Swal.fire({
    icon: 'success',
    title: titulo,
    text: mensaje,
    toast: true,
    position: 'top-end',
    showConfirmButton: false,
    timer: 3000,
    timerProgressBar: true
  });
}

/**
 * Mostrar alerta de error
 */
function mostrarError(titulo, mensaje = '') {
  Swal.fire({
    icon: 'error',
    title: titulo,
    text: mensaje,
    confirmButtonText: 'Entendido'
  });
}

/**
 * Cargar marcas disponibles
 */
async function cargarMarcas(selectId = 'marcaMoto') {
  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const response = await fetch('http://localhost:3001/api/marcas', {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (response.ok) {
      const marcas = await response.json();
      const select = document.getElementById(selectId);
      
      if (select) {
        select.innerHTML = '<option value="">Seleccionar marca...</option>';
        
        marcas.forEach(marca => {
          const option = document.createElement('option');
          option.value = marca.id;
          option.textContent = marca.nombre;
          select.appendChild(option);
        });
        
        console.log('✅ Marcas cargadas correctamente');
        return marcas;
      }
    } else {
      console.error('Error cargando marcas:', response.status);
      return [];
    }
  } catch (error) {
    console.error('Error cargando marcas:', error);
    return [];
  }
}

/**
 * Cargar modelos por marca
 */
async function cargarModelos(marcaId, selectId = 'modeloMoto') {
  const select = document.getElementById(selectId);
  
  if (!select) {
    console.error('Select de modelos no encontrado:', selectId);
    return [];
  }

  select.innerHTML = '<option value="">Cargando modelos...</option>';
  select.disabled = true;

  if (!marcaId) {
    select.innerHTML = '<option value="">Seleccionar modelo...</option>';
    select.disabled = false;
    return [];
  }

  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const response = await fetch(`http://localhost:3001/api/modelos?marca=${marcaId}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (response.ok) {
      const modelos = await response.json();
      
      select.innerHTML = '<option value="">Seleccionar modelo...</option>';
      
      if (modelos.length === 0) {
        const modelosGenericos = obtenerModelosGenericos(marcaId);
        modelosGenericos.forEach(modelo => {
          const option = document.createElement('option');
          option.value = `generico_${modelo}`;
          option.textContent = modelo;
          option.setAttribute('data-generico', 'true');
          select.appendChild(option);
        });
        console.log('⚠️ No se encontraron modelos en BD, cargando genéricos');
      } else {
        modelos.forEach(modelo => {
          const option = document.createElement('option');
          option.value = modelo.id;
          option.textContent = modelo.nombre;
          select.appendChild(option);
        });
        console.log('✅ Modelos cargados correctamente');
      }
      
      select.disabled = false;
      return modelos;
    } else {
      console.error('Error cargando modelos:', response.status);
      select.innerHTML = '<option value="">Seleccionar modelo...</option>';
      const modelosGenericos = obtenerModelosGenericos(marcaId);
      modelosGenericos.forEach(modelo => {
        const option = document.createElement('option');
        option.value = `generico_${modelo}`;
        option.textContent = modelo;
        option.setAttribute('data-generico', 'true');
        select.appendChild(option);
      });
      select.disabled = false;
      console.log('✅ Modelos genéricos cargados como fallback');
      return [];
    }
  } catch (error) {
    console.error('Error cargando modelos:', error);
    select.innerHTML = '<option value="">Error cargando modelos</option>';
    select.disabled = false;
    return [];
  }
}

/**
 * Obtener modelos genéricos por marca
 */
function obtenerModelosGenericos(marcaId) {
  const modelosPorMarca = {
    1: ['YZF-R1', 'YZF-R6', 'MT-07', 'MT-09', 'FZ-25', 'XTZ 250'],
    2: ['CBR1000RR', 'CBR600RR', 'CB650R', 'CB300R', 'CRF450L', 'PCX 150'],
    3: ['Ninja ZX-10R', 'Ninja 650', 'Z400', 'Versys 650', 'KLX 230'],
    4: ['S1000RR', 'F850GS', 'G310R', 'R1250GS'],
    5: ['Panigale V4', 'Monster 821', 'Scrambler Icon'],
    6: ['Speed Triple', 'Street Triple', 'Tiger 900'],
    7: ['V-Strom 650', 'GSX-R750', 'Boulevard M109R'],
    8: ['RC 390', 'Duke 390', 'Adventure 390']
  };
  
  return modelosPorMarca[marcaId] || ['Modelo Genérico', 'Otro Modelo'];
}

/**
 * Cargar regiones
 */
async function cargarRegiones(selectId = 'clienteRegion') {
  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const response = await fetch('http://localhost:3001/api/regiones', {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (response.ok) {
      const regiones = await response.json();
      const select = document.getElementById(selectId);
      
      if (select) {
        select.innerHTML = '<option value="">Seleccionar región...</option>';
        
        regiones.forEach(region => {
          const option = document.createElement('option');
          option.value = region.id;
          option.textContent = region.nombre;
          select.appendChild(option);
        });
        
        console.log('✅ Regiones cargadas correctamente');
        return regiones;
      }
    } else {
      console.error('Error cargando regiones:', response.status);
      return [];
    }
  } catch (error) {
    console.error('Error cargando regiones:', error);
    return [];
  }
}

/**
 * Cargar tipos de mantención
 */
async function cargarTiposMantencion() {
  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const response = await fetch('http://localhost:3001/api/tipos-mantencion', {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (response.ok) {
      const tipos = await response.json();
      console.log('✅ Tipos de mantención cargados');
      return tipos;
    } else {
      // Datos por defecto si falla la API
      return [
        { id: 1, nombre: 'Mantenimiento Básico', descripcion: 'Aceite, filtros básicos' },
        { id: 2, nombre: 'Mantenimiento Mayor', descripcion: 'Revisión completa de motor' },
        { id: 3, nombre: 'Reparación', descripcion: 'Reparación específica' },
        { id: 4, nombre: 'Carrocería', descripcion: 'Trabajos de pintura/chapa' }
      ];
    }
  } catch (error) {
    console.error('Error cargando tipos de mantención:', error);
    return [
      { id: 1, nombre: 'Mantenimiento Básico', descripcion: 'Aceite, filtros básicos' },
      { id: 2, nombre: 'Mantenimiento Mayor', descripcion: 'Revisión completa de motor' },
      { id: 3, nombre: 'Reparación', descripcion: 'Reparación específica' },
      { id: 4, nombre: 'Carrocería', descripcion: 'Trabajos de pintura/chapa' }
    ];
  }
}

/**
 * Cargar tipos de equipamiento
 */
async function cargarTiposEquipamiento() {
  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const response = await fetch('http://localhost:3001/api/tipos-equipamiento', {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (response.ok) {
      const tipos = await response.json();
      console.log('✅ Tipos de equipamiento cargados');
      return tipos;
    } else {
      // Datos por defecto si falla la API
      return [
        { id: 1, categoria: 'Seguridad', nombre: 'Alarma' },
        { id: 2, categoria: 'Confort', nombre: 'Baúl trasero' },
        { id: 3, categoria: 'Estética', nombre: 'Kit de luces LED' },
        { id: 4, categoria: 'Performance', nombre: 'Escape deportivo' },
        { id: 5, categoria: 'Protección', nombre: 'Protector de motor' }
      ];
    }
  } catch (error) {
    console.error('Error cargando tipos de equipamiento:', error);
    return [
      { id: 1, categoria: 'Seguridad', nombre: 'Alarma' },
      { id: 2, categoria: 'Confort', nombre: 'Baúl trasero' },
      { id: 3, categoria: 'Estética', nombre: 'Kit de luces LED' },
      { id: 4, categoria: 'Performance', nombre: 'Escape deportivo' },
      { id: 5, categoria: 'Protección', nombre: 'Protector de motor' }
    ];
  }
}

/**
 * Cargar trabajadores de la empresa
 */
async function cargarTrabajadores() {
  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const id_empresa = localStorage.getItem('id_empresa');
    
    const response = await fetch(`http://localhost:3001/api/trabajadores?empresa=${id_empresa}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (response.ok) {
      const trabajadores = await response.json();
      console.log('✅ Trabajadores cargados');
      return trabajadores;
    } else {
      console.error('Error cargando trabajadores:', response.status);
      return [];
    }
  } catch (error) {
    console.error('Error cargando trabajadores:', error);
    return [];
  }
}

/**
 * Configurar fecha por defecto (hoy)
 */
function configurarFechaDefecto(campoId = 'fechaIngreso') {
  const campo = document.getElementById(campoId);
  if (campo && !campo.value) {
    const hoy = new Date().toISOString().split('T')[0];
    campo.value = hoy;
  }
}

/**
 * Cargar datos del usuario desde localStorage
 */
function cargarDatosUsuario() {
  try {
    const datosUsuario = {
      id_empresa: localStorage.getItem('id_empresa') || '1',
      id_usuario: localStorage.getItem('id_usuario') || '1',
      rol: localStorage.getItem('rol') || 'usuario',
      usuario: JSON.parse(localStorage.getItem('usuario') || '{}')
    };
    
    console.log('👤 Datos usuario cargados:', datosUsuario);
    return datosUsuario;
    
  } catch (error) {
    console.error('❌ Error cargando datos de usuario:', error);
    return { 
      id_empresa: '1', 
      id_usuario: '1', 
      rol: 'usuario',
      usuario: {}
    };
  }
}

/**
 * Obtener cotización del dólar (opcional)
 */
async function obtenerCotizacionDolar() {
  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const response = await fetch('http://localhost:3001/api/cotizacion-dolar', {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (response.ok) {
      const cotizacion = await response.json();
      return cotizacion?.valor || 900;
    } else {
      return 900; // Valor por defecto
    }
  } catch (error) {
    console.warn('No se pudo obtener cotización del dólar:', error);
    return 900;
  }
}

/**
 * Configurar tema (light/dark) con toggle
 */
function configurarTema() {
  const savedTheme = localStorage.getItem('theme') || 'light';
  document.documentElement.classList.toggle('dark', savedTheme === 'dark');
  
  const themeToggle = document.getElementById('themeToggle');
  const themeIcon = document.getElementById('themeIcon');
  
  if (themeToggle) {
    themeToggle.addEventListener('click', function() {
      document.documentElement.classList.toggle('dark');
      const isDark = document.documentElement.classList.contains('dark');
      localStorage.setItem('theme', isDark ? 'dark' : 'light');
      
      // Actualizar icono si existe
      if (themeIcon) {
        themeIcon.innerHTML = isDark 
          ? '<path fill-rule="evenodd" d="M17.293 13.293A8 8 0 016.707 2.707a8.001 8.001 0 1010.586 10.586z" clip-rule="evenodd"></path>'
          : '<path fill-rule="evenodd" d="M10 2a1 1 0 011 1v1a1 1 0 11-2 0V3a1 1 0 011-1zm4 8a4 4 0 11-8 0 4 4 0 018 0zm-.464 4.95l.707.707a1 1 0 001.414-1.414l-.707-.707a1 1 0 00-1.414 1.414zm2.12-10.607a1 1 0 010 1.414l-.706.707a1 1 0 11-1.414-1.414l.707-.707a1 1 0 011.414 0zM17 11a1 1 0 100-2h-1a1 1 0 100 2h1zm-7 4a1 1 0 011 1v1a1 1 0 11-2 0v-1a1 1 0 011-1zM5.05 6.464A1 1 0 106.465 5.05l-.708-.707a1 1 0 00-1.414 1.414l.707.707zm1.414 8.486l-.707.707a1 1 0 01-1.414-1.414l.707-.707a1 1 0 011.414 1.414zM4 11a1 1 0 100-2H3a1 1 0 000 2h1z" clip-rule="evenodd"></path>';
      }
    });
  }
}

/**
 * Función genérica para realizar peticiones API
 */
async function apiRequest(endpoint, options = {}) {
  const defaultOptions = {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${localStorage.getItem('authToken') || localStorage.getItem('token')}`
    }
  };

  const config = { ...defaultOptions, ...options };
  const url = endpoint.startsWith('http') ? endpoint : `http://localhost:3001/api${endpoint}`;

  try {
    const response = await fetch(url, config);
    
    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    
    const contentType = response.headers.get('content-type');
    if (contentType && contentType.includes('application/json')) {
      return await response.json();
    } else {
      return await response.text();
    }
  } catch (error) {
    console.error('Error en petición API:', error);
    throw error;
  }
}

// ====================================================================
// INICIALIZACIÓN AUTOMÁTICA
// ====================================================================

document.addEventListener('DOMContentLoaded', function() {
  if (window.location.pathname.includes('/operaciones/')) {
    console.log('🚀 Global.js inicializado para operaciones');
    
    if (!verificarAutenticacion()) {
      return;
    }
    
    obtenerTrabajadorActual();
  }
});

console.log('📦 Operaciones Global.js cargado correctamente');

// ====================================================================
// GESTIÓN DE MOTOS
// ====================================================================

// Reusable function to display a list of motorcycles
function mostrarMotosEnLista(motos, containerId = 'listaMotosVenta') {
  const container = document.getElementById(containerId);

  if (!container) {
    console.error('Container not found:', containerId);
    return;
  }

  if (!motos || motos.length === 0) {
    container.innerHTML = `
      <div class="text-center py-8 text-gray-500 dark:text-gray-400">
        <div class="text-4xl mb-2">🏍️</div>
        <p>No hay motos disponibles para la venta</p>
      </div>
    `;
    return;
  }

  container.innerHTML = motos.map(moto => `
    <div class="moto-card bg-white dark:bg-gray-700 border border-gray-200 dark:border-gray-600 rounded-lg p-4 hover:shadow-md transition-all cursor-pointer">
      <div class="flex justify-between items-start mb-3">
        <div>
          <h4 class="font-semibold text-gray-900 dark:text-white">${moto.marca} ${moto.modelo}</h4>
          <p class="text-sm text-gray-600 dark:text-gray-400">Año: ${moto.ano || 'N/A'}</p>
        </div>
        <div class="text-right">
          <div class="text-lg font-bold text-green-600 dark:text-green-400">
            $${(moto.precio_venta || moto.precio || 0).toLocaleString('es-CL')}
          </div>
          <div class="text-xs text-gray-500">Precio venta</div>
        </div>
      </div>
    </div>
  `).join('');
}

// ====================================================================
// DETALLES DE VENTA
// ====================================================================

// Reusable function to display sale details
function mostrarDetallesVenta(motoSeleccionada, detallesDivId = 'detallesVenta', precioInputId = 'precioVenta') {
  const detallesDiv = document.getElementById(detallesDivId);
  if (!detallesDiv) {
    console.error('Details container not found:', detallesDivId);
    return;
  }

  detallesDiv.classList.remove('hidden');

  const precioInput = document.getElementById(precioInputId);
  if (precioInput && motoSeleccionada) {
    precioInput.value = motoSeleccionada.precio_venta || motoSeleccionada.precio || '';
  }

  detallesDiv.scrollIntoView({ behavior: 'smooth', block: 'center' });
}

// ====================================================================
// FORMULARIOS
// ====================================================================

// Reusable function to reset the sale form
function reiniciarFormulario(formFields = ['precioVenta', 'descuentoVenta', 'metodoPago', 'observacionesVenta'], hiddenSections = ['infoClienteSeleccionado', 'infoMotoSeleccionada', 'detallesVenta']) {
  formFields.forEach(fieldId => {
    const field = document.getElementById(fieldId);
    if (field) field.value = '';
  });

  hiddenSections.forEach(sectionId => {
    const section = document.getElementById(sectionId);
    if (section) section.classList.add('hidden');
  });

  console.log('Formulario reiniciado');
}
