/**
 * OPERACIONES DE VENTA - JavaScript específico
 * ====================================================================
 */

// Variables específicas del módulo de venta
let pasoActual = 1;
let clienteVenta = null;  // Renamed to avoid conflict with global.js
let motoVenta = null;     // Renamed to avoid conflict with global.js
let motosDisponibles = [];
let huellaAutenticada = false;
let tipoOperacion = null; // Nueva variable para el tipo de operación
let empresaActual = null; // Información de la empresa actual
let vendedorActual = null; // Información del vendedor actual

// ====================================================================
// INICIALIZACIÓN
// ====================================================================

document.addEventListener('DOMContentLoaded', function() {
  console.log('🚀 Operaciones de Venta inicializado');
  
  // Cargar datos iniciales
  cargarInformacionEmpresa();
  cargarInformacionVendedor();
  cargarClientes();
  cargarRegiones();
  cargarMotosDisponibles();
  
  // Configurar eventos
  configurarEventos();
  
  // Activar el primer tab
  cambiarTab('vender');
});

function configurarEventos() {
  // Ocultar resultados de búsqueda al hacer clic fuera
  document.addEventListener('click', function(e) {
    if (!e.target.closest('#buscarCliente') && !e.target.closest('#resultadosClientes')) {
      document.getElementById('resultadosClientes').classList.add('hidden');
    }
  });
}

// ====================================================================
// GESTIÓN DE PASOS
// ====================================================================

function continuarAPaso(nuevoPaso) {
  console.log(`🔄 Intentando navegar del paso ${pasoActual} al paso ${nuevoPaso}`);
  
  // Validar que se puede continuar al paso solicitado
  if (!validarPasoActual()) {
    console.log('❌ Validación fallida, no se puede continuar');
    return;
  }
  
  // Marcar paso actual como completado
  if (pasoActual > 0) {
    const stepActual = document.getElementById(`step-${pasoActual}`);
    if (stepActual) {
      stepActual.classList.add('completed');
      stepActual.classList.remove('active');
    }
  }
  
  // Ocultar paso actual
  const pasoActualDiv = document.getElementById(`paso-${pasoActual}`);
  if (pasoActualDiv) {
    pasoActualDiv.classList.remove('active');
  }
  
  // Mostrar nuevo paso
  pasoActual = nuevoPaso;
  const nuevoPasoDiv = document.getElementById(`paso-${pasoActual}`);
  const nuevoStep = document.getElementById(`step-${pasoActual}`);
  
  if (nuevoPasoDiv) {
    nuevoPasoDiv.classList.add('active');
  }
  if (nuevoStep) {
    nuevoStep.classList.add('active');
  }
  
  // Ejecutar acciones específicas del paso
  switch(pasoActual) {
    case 1:
      console.log('📍 Paso 1: Selección de cliente');
      break;
    case 2:
      console.log('📍 Paso 2: Selección de moto');
      cargarMotosDisponibles();
      break;
    case 3:
      console.log('📍 Paso 3: Resumen de venta');
      mostrarResumenVenta();
      break;
    case 4:
      console.log('📍 Paso 4: Autenticación biométrica');
      prepararAutenticacion();
      break;
  }
  
  console.log(`📍 Navegando al paso ${pasoActual}`);
}

function validarPasoActual() {
  switch(pasoActual) {
    case 1:
      if (!clienteVenta) {
        console.log('❌ Cliente no seleccionado');
        if (typeof mostrarError === 'function') {
          mostrarError('Cliente requerido', 'Debes seleccionar un cliente antes de continuar');
        } else {
          alert('Debes seleccionar un cliente antes de continuar');
        }
        return false;
      }
      break;
    case 2:
      if (!motoVenta) {
        console.log('❌ Moto no seleccionada');
        if (typeof mostrarError === 'function') {
          mostrarError('Moto requerida', 'Debes seleccionar una moto antes de continuar');
        } else {
          alert('Debes seleccionar una moto antes de continuar');
        }
        return false;
      }
      break;
    case 3:
      const precioInput = document.getElementById('precioVenta');
      const precio = precioInput ? precioInput.value : '';
      if (!precio || parseFloat(precio) <= 0) {
        console.log('❌ Precio no válido');
        if (typeof mostrarError === 'function') {
          mostrarError('Precio requerido', 'Debes establecer un precio de venta válido');
        } else {
          alert('Debes establecer un precio de venta válido');
        }
        return false;
      }
      // Validar que se haya seleccionado el tipo de operación
      if (!tipoOperacion) {
        console.log('❌ Tipo de operación no seleccionado');
        document.getElementById('alerta-seleccion').classList.remove('hidden');
        if (typeof mostrarError === 'function') {
          mostrarError('Tipo de operación requerido', 'Debes seleccionar si es una reserva o venta directa');
        } else {
          alert('Debes seleccionar si es una reserva o venta directa');
        }
        return false;
      }
      break;
  }
  return true;
}

// ====================================================================
// GESTIÓN DE TIPO DE OPERACIÓN
// ====================================================================

function seleccionarTipoOperacion(tipo) {
  console.log('🔄 Seleccionando tipo de operación:', tipo);
  
  tipoOperacion = tipo;
  
  // Ocultar alerta de selección
  document.getElementById('alerta-seleccion').classList.add('hidden');
  
  // Resetear todas las opciones
  document.getElementById('opcion-reservar').classList.remove('border-orange-400', 'bg-orange-50');
  document.getElementById('opcion-reservar').classList.add('border-orange-200');
  document.getElementById('dot-reservar').classList.add('hidden');
  
  document.getElementById('opcion-venta-directa').classList.remove('border-green-400', 'bg-green-50');
  document.getElementById('opcion-venta-directa').classList.add('border-green-200');
  document.getElementById('dot-venta-directa').classList.add('hidden');
  
  // Activar la opción seleccionada
  if (tipo === 'RESERVAR') {
    document.getElementById('opcion-reservar').classList.remove('border-orange-200');
    document.getElementById('opcion-reservar').classList.add('border-orange-400', 'bg-orange-50');
    document.getElementById('dot-reservar').classList.remove('hidden');
    
    // Actualizar texto del botón
    document.getElementById('texto-continuar').textContent = 'Generar Nota de Venta';
  } else if (tipo === 'VENTA_DIRECTA') {
    document.getElementById('opcion-venta-directa').classList.remove('border-green-200');
    document.getElementById('opcion-venta-directa').classList.add('border-green-400', 'bg-green-50');
    document.getElementById('dot-venta-directa').classList.remove('hidden');
    
    // Actualizar texto del botón
    document.getElementById('texto-continuar').textContent = 'Proceder a Autenticación';
  }
  
  console.log('✅ Tipo de operación seleccionado:', tipoOperacion);
}

function validarYContinuar() {
  console.log('🔄 Validando tipo de operación antes de continuar');
  
  if (!tipoOperacion) {
    console.log('❌ No se ha seleccionado tipo de operación');
    document.getElementById('alerta-seleccion').classList.remove('hidden');
    
    if (typeof mostrarError === 'function') {
      mostrarError('Selección requerida', 'Debes seleccionar el tipo de operación antes de continuar');
    } else {
      alert('Debes seleccionar el tipo de operación antes de continuar');
    }
    return;
  }
  
  console.log('✅ Tipo de operación válido:', tipoOperacion);
  
  // Continuar según el tipo de operación
  if (tipoOperacion === 'RESERVAR') {
    // Para reservas, ir directo a generación de documento (saltamos autenticación)
    procesarReserva();
  } else if (tipoOperacion === 'VENTA_DIRECTA') {
    // Para venta directa, ir a autenticación biométrica
    continuarAPaso(4);
  }
}

// ====================================================================
// FUNCIONES DE BÚSQUEDA DE CLIENTES
// ====================================================================

// ====================================================================
// BÚSQUEDA Y FORMATEO DE CLIENTES
// ====================================================================

function manejarBusquedaConFormato(input) {
  const valor = input.value;
  console.log('🔤 Valor original:', valor);
  
  // Detectar si es potencialmente un RUT (solo números y k/K)
  const esRUT = /^[0-9kK\-\.]+$/.test(valor) && valor.length > 0;
  
  if (esRUT && valor.length > 1) {
    // Aplicar formateo de RUT en tiempo real
    const valorFormateado = formatearRUTInput(valor);
    input.value = valorFormateado;
    console.log('✅ RUT formateado:', valorFormateado);
    
    // Buscar con el valor formateado
    buscarClientesEnTiempoReal(valorFormateado);
  } else {
    // Es texto normal (nombre), buscar directamente
    console.log('📝 Búsqueda por nombre:', valor);
    buscarClientesEnTiempoReal(valor);
  }
}

function formatearRUTInput(valor) {
  // Limpiar el valor: solo números y K
  let rut = valor.replace(/[^0-9kK]/g, '');
  
  if (rut.length === 0) return '';
  
  // Si tiene más de 1 dígito, separar cuerpo y dígito verificador
  if (rut.length > 1) {
    const cuerpo = rut.slice(0, -1);
    const dv = rut.slice(-1);
    
    // Formatear el cuerpo con puntos cada 3 dígitos desde la derecha
    const cuerpoFormateado = cuerpo.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
    
    // Retornar con guión antes del DV
    return cuerpoFormateado + '-' + dv;
  }
  
  return rut;
}

function buscarClientesEnTiempoReal(termino) {
  const resultadosDiv = document.getElementById('resultadosClientes');
  const clienteNoEncontrado = document.getElementById('clienteNoEncontrado');
  
  if (!termino || termino.length < 2) {
    resultadosDiv.classList.add('hidden');
    if (clienteNoEncontrado) {
      clienteNoEncontrado.classList.add('hidden');
    }
    return;
  }

  // Simular búsqueda API (aquí integrarías con el backend)
  setTimeout(() => {
    buscarClientesLocal(termino);
  }, 300);
}

async function buscarClientesLocal(termino) {
  try {
    console.log('🔍 Buscando clientes con término:', termino);
    
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const id_empresa = localStorage.getItem('id_empresa');
    
    if (!token || !id_empresa) {
      throw new Error('No hay sesión activa');
    }
    
    const response = await fetch(`http://localhost:3001/api/clientes/buscar?termino=${encodeURIComponent(termino)}&id_empresa=${id_empresa}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });
    
    if (!response.ok) {
      throw new Error(`Error ${response.status}: ${response.statusText}`);
    }
    
    const clientes = await response.json();
    mostrarResultadosBusqueda(clientes);
    
  } catch (error) {
    console.error('❌ Error al buscar clientes:', error);
    // Fallback: buscar en datos locales si hay algún problema con la API
    mostrarResultadosBusqueda([]);
  }
}

function mostrarResultadosBusqueda(clientes) {
  const resultadosDiv = document.getElementById('resultadosClientes');
  const clienteNoEncontrado = document.getElementById('clienteNoEncontrado');
  
  if (!clientes || clientes.length === 0) {
    // NO hay resultados - mostrar solo el mensaje "no encontrado"
    resultadosDiv.classList.add('hidden');
    clienteNoEncontrado.classList.remove('hidden');
  } else {
    // SÍ hay resultados - mostrar solo los clientes encontrados
    clienteNoEncontrado.classList.add('hidden');
    resultadosDiv.innerHTML = clientes.map(cliente => `
      <div class="bg-white dark:bg-gray-700 border border-gray-200 dark:border-gray-600 rounded-lg p-3 hover:bg-gray-50 dark:hover:bg-gray-600 cursor-pointer transition-all" 
           onclick="seleccionarCliente(${JSON.stringify(cliente).replace(/"/g, '&quot;')})">
        <div class="flex items-center justify-between">
          <div class="flex items-center space-x-3">
            <div class="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center text-white text-sm font-semibold">
              ${cliente.nombres ? cliente.nombres.charAt(0).toUpperCase() : 'C'}
            </div>
            <div>
              <p class="font-medium text-gray-900 dark:text-white text-sm">
                ${cliente.nombres || ''} ${cliente.apellidos || ''}
              </p>
              <p class="text-xs text-gray-600 dark:text-gray-400">
                RUT: ${cliente.cedula || 'No especificado'}
              </p>
            </div>
          </div>
          ${cliente.telefono ? `
            <p class="text-xs text-gray-500 dark:text-gray-400">
              📞 ${cliente.telefono}
            </p>
          ` : ''}
        </div>
      </div>
    `).join('');
    resultadosDiv.classList.remove('hidden');
  }
}

function seleccionarCliente(cliente) {
  clienteVenta = cliente;
  console.log('🔄 Seleccionando cliente:', cliente);
  
  // Asegurar que los campos existan con valores por defecto
  const nombres = cliente.nombres || '';
  const apellidos = cliente.apellidos || '';
  const cedula = cliente.cedula || 'No especificado';
  const telefono = cliente.telefono || '';
  
  // Mostrar información del cliente seleccionado
  const infoDiv = document.getElementById('infoClienteSeleccionado');
  if (infoDiv) {
    infoDiv.innerHTML = `
      <div>
        <p class="font-medium">${nombres} ${apellidos}</p>
        <p class="text-sm">RUT: ${cedula}</p>
        ${telefono ? `<p class="text-sm">📞 ${telefono}</p>` : ''}
      </div>
    `;
  }
  
  // Mostrar contenedor del cliente seleccionado
  const contenedorCliente = document.getElementById('clienteSeleccionado');
  if (contenedorCliente) {
    contenedorCliente.classList.remove('hidden');
  }
  
  // Ocultar resultados de búsqueda
  const resultados = document.getElementById('resultadosClientes');
  if (resultados) {
    resultados.classList.add('hidden');
  }
  
  // Habilitar botón de continuar
  const btnContinuar = document.getElementById('btn-continuar-paso1');
  if (btnContinuar) {
    btnContinuar.disabled = false;
  }
  
  console.log('✅ Cliente seleccionado:', cliente);
}

function limpiarSeleccionCliente() {
  clienteVenta = null;
  
  const buscarInput = document.getElementById('buscarCliente');
  if (buscarInput) {
    buscarInput.value = '';
  }
  
  const contenedorCliente = document.getElementById('clienteSeleccionado');
  if (contenedorCliente) {
    contenedorCliente.classList.add('hidden');
  }
  
  const resultados = document.getElementById('resultadosClientes');
  if (resultados) {
    resultados.classList.add('hidden');
  }
  
  const clienteNoEncontrado = document.getElementById('clienteNoEncontrado');
  if (clienteNoEncontrado) {
    clienteNoEncontrado.classList.add('hidden');
  }
  
  const btnContinuar = document.getElementById('btn-continuar-paso1');
  if (btnContinuar) {
    btnContinuar.disabled = true;
  }
}

// ====================================================================
// REGISTRO DE NUEVOS CLIENTES
// ====================================================================

function mostrarFormularioRegistro() {
  const formulario = document.getElementById('formularioRegistroCliente');
  const clienteNoEncontrado = document.getElementById('clienteNoEncontrado');
  
  if (formulario) {
    formulario.classList.remove('hidden');
  }
  if (clienteNoEncontrado) {
    clienteNoEncontrado.classList.add('hidden');
  }
}

function ocultarFormularioRegistro() {
  const formulario = document.getElementById('formularioRegistroCliente');
  const form = document.getElementById('formRegistroCliente');
  
  if (formulario) {
    formulario.classList.add('hidden');
  }
  if (form) {
    form.reset();
  }
}

async function registrarCliente(event) {
  event.preventDefault();
  
  const form = event.target;
  const formData = new FormData(form);
  
  // Debug: mostrar todos los datos del formulario
  console.log('📋 Datos del formulario:');
  for (let [key, value] of formData.entries()) {
    console.log(`  ${key}: ${value}`);
  }
  
  // Validar campos requeridos
  const nombresRequired = ['nombres', 'apellidos', 'cedula', 'tipo_cliente', 'edad', 'telefono', 'id_region'];
  for (const field of nombresRequired) {
    if (!formData.get(field)) {
      mostrarNotificacion(`El campo ${field} es requerido`, 'error');
      return;
    }
  }
  
  // Validar edad
  const edad = parseInt(formData.get('edad'));
  if (edad < 18 || edad > 120) {
    mostrarNotificacion('La edad debe ser entre 18 y 120 años', 'error');
    return;
  }
  
  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const id_empresa = localStorage.getItem('id_empresa');
    
    if (!token || !id_empresa) {
      throw new Error('No hay sesión activa');
    }
    
    const clienteData = {
      nombres: formData.get('nombres'),
      apellidos: formData.get('apellidos'),
      cedula: formData.get('cedula'),
      tipo_cliente: formData.get('tipo_cliente'),
      edad: parseInt(formData.get('edad')),
      telefono: formData.get('telefono'),
      email: formData.get('email') || null,
      id_region: parseInt(formData.get('id_region')),
      direccion: formData.get('direccion') || null,
      id_empresa: parseInt(id_empresa)
    };
    
    console.log('📤 Enviando datos del cliente:', clienteData);
    
    const response = await fetch('http://localhost:3001/api/clientes', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(clienteData)
    });
    
    if (!response.ok) {
      const errorData = await response.json();
      throw new Error(errorData.message || 'Error al registrar cliente');
    }
    
    const nuevoCliente = await response.json();
    console.log('✅ Cliente registrado:', nuevoCliente);
    
    // Crear objeto cliente con la estructura correcta para la selección
    const clienteParaSeleccion = {
      id: nuevoCliente.id || nuevoCliente.insertId,
      nombres: clienteData.nombres,
      apellidos: clienteData.apellidos,
      cedula: clienteData.cedula,
      telefono: clienteData.telefono,
      email: clienteData.email,
      tipo_cliente: clienteData.tipo_cliente,
      edad: clienteData.edad,
      id_region: clienteData.id_region,
      direccion: clienteData.direccion
    };
    
    // Seleccionar automáticamente el cliente recién registrado
    seleccionarCliente(clienteParaSeleccion);
    
    // Ocultar formulario y limpiar
    ocultarFormularioRegistro();
    
    // Limpiar campo de búsqueda
    const buscarInput = document.getElementById('buscarCliente');
    if (buscarInput) {
      buscarInput.value = '';
    }
    
    mostrarNotificacion('Cliente registrado exitosamente', 'success');
    
  } catch (error) {
    console.error('❌ Error al registrar cliente:', error);
    mostrarNotificacion(error.message || 'Error al registrar cliente', 'error');
  }
}

// Funciones de validación y formateo
function formatearRUT(input) {
  // Usar la función de formateo unificada
  input.value = formatearRUTInput(input.value);
}

function formatearTelefono(input) {
  let value = input.value.replace(/[^\d+]/g, '');
  
  // Si no empieza con +56, agregarlo para Chile
  if (!value.startsWith('+56') && value.length > 0) {
    if (value.startsWith('56')) {
      value = '+' + value;
    } else {
      value = '+56' + value;
    }
  }
  
  input.value = value;
}

function mostrarNotificacion(mensaje, tipo = 'info') {
  // Crear notificación temporal
  const notificacion = document.createElement('div');
  notificacion.className = `fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg text-white transition-all duration-300 ${
    tipo === 'success' ? 'bg-green-500' : 
    tipo === 'error' ? 'bg-red-500' : 
    'bg-blue-500'
  }`;
  notificacion.textContent = mensaje;
  
  document.body.appendChild(notificacion);
  
  // Remover después de 3 segundos
  setTimeout(() => {
    notificacion.remove();
  }, 3000);
}

function volverAlDashboard() {
  // Navegación relativa de vuelta al dashboard (puerto 3000)
  window.location.href = '../consignar-vender.html';
}

// ====================================================================
// FILTROS MEJORADOS
// ====================================================================

// Variables para filtros
let filtrosActivos = {
  condicion: 'todas',
  estado: 'todas',
  marca: '',
  precio: '',
  ano: '',
  busqueda: ''
};

function aplicarFiltro(tipo, valor) {
  // Actualizar filtro activo
  filtrosActivos[tipo] = valor;
  
  // Actualizar UI de botones
  actualizarBotonesFiltro(tipo, valor);
  
  // Aplicar filtros a las motos
  filtrarMotosConFiltros();
}

function actualizarBotonesFiltro(tipo, valor) {
  // Remover clase activa de todos los botones del tipo
  const botones = document.querySelectorAll(`.filtro-btn[onclick*="${tipo}"]`);
  botones.forEach(btn => {
    btn.classList.remove('filtro-activo');
  });
  
  // Agregar clase activa al botón seleccionado
  const btnActivo = document.querySelector(`.filtro-btn[onclick*="${tipo}, '${valor}'"]`);
  if (btnActivo) {
    btnActivo.classList.add('filtro-activo');
  }
}

function filtrarMotosConFiltros() {
  const todasLasMotos = document.querySelectorAll('#listaMotosVenta .moto-card');
  let motosVisibles = 0;
  
  todasLasMotos.forEach(motoCard => {
    let mostrar = true;
    
    // Obtener datos de la moto desde los atributos de datos
    const marca = motoCard.dataset.marca || '';
    const modelo = motoCard.dataset.modelo || '';
    const estado = motoCard.dataset.estado || '';
    const condicion = motoCard.dataset.condicion || '';
    const patente = motoCard.dataset.patente || '';
    const precio = parseInt(motoCard.dataset.precio) || 0;
    const ano = parseInt(motoCard.dataset.ano) || 0;
    const textoCompleto = motoCard.textContent.toLowerCase();
    
    // Filtro por estado
    if (filtrosActivos.estado !== 'todas') {
      if (estado.toLowerCase() !== filtrosActivos.estado.toLowerCase()) {
        mostrar = false;
      }
    } else {
      // Si no hay filtro específico, mostrar solo DISPONIBLES y RESERVADAS
      if (estado !== 'DISPONIBLE' && estado !== 'RESERVADA') {
        mostrar = false;
      }
    }
    
    // Filtro por condición
    if (filtrosActivos.condicion !== 'todas') {
      if (condicion.toLowerCase() !== filtrosActivos.condicion.toLowerCase()) {
        mostrar = false;
      }
    }
    
    // Filtro por marca
    if (filtrosActivos.marca && marca.toLowerCase() !== filtrosActivos.marca.toLowerCase()) {
      mostrar = false;
    }
    
    // Filtro por precio
    if (filtrosActivos.precio && precio > parseInt(filtrosActivos.precio)) {
      mostrar = false;
    }
    
    // Filtro por año
    if (filtrosActivos.ano && ano < parseInt(filtrosActivos.ano)) {
      mostrar = false;
    }
    
    // Filtro por búsqueda de texto
    if (filtrosActivos.busqueda) {
      const termino = filtrosActivos.busqueda.toLowerCase();
      // Buscar en campos específicos y en todo el texto de la tarjeta
      const camposBusqueda = [
        marca.toLowerCase(),
        modelo.toLowerCase(),
        patente.toLowerCase(),
        textoCompleto
      ].join(' ');
      
      if (!camposBusqueda.includes(termino)) {
        mostrar = false;
      }
    }
    
    // Mostrar u ocultar moto
    if (mostrar) {
      motoCard.classList.remove('filtered-out');
      motoCard.style.display = 'block';
      motoCard.classList.add('highlight');
      motosVisibles++;
      
      // Remover highlight después de la animación
      setTimeout(() => {
        motoCard.classList.remove('highlight');
      }, 500);
    } else {
      motoCard.classList.add('filtered-out');
      motoCard.style.display = 'none';
    }
  });
  
  // Actualizar contador
  actualizarContadorResultados(motosVisibles);
  
  // Mostrar mensaje si no hay resultados
  const container = document.getElementById('listaMotosVenta');
  const mensajeVacio = container.querySelector('.mensaje-sin-resultados');
  
  if (motosVisibles === 0 && todasLasMotos.length > 0) {
    if (!mensajeVacio) {
      const divMensaje = document.createElement('div');
      divMensaje.className = 'mensaje-sin-resultados col-span-full text-center py-8 text-gray-500 dark:text-gray-400';
      divMensaje.innerHTML = `
        <div class="text-4xl mb-2">🔍</div>
        <p class="text-lg font-medium mb-2">No se encontraron motocicletas</p>
        <p class="text-sm">Intenta ajustar los filtros de búsqueda</p>
        <button onclick="limpiarFiltros()" class="mt-4 bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded-lg transition-colors">
          Limpiar Filtros
        </button>
      `;
      container.appendChild(divMensaje);
    }
  } else if (mensajeVacio) {
    mensajeVacio.remove();
  }
}

function extraerPrecio(motoCard) {
  const textoCard = motoCard.textContent;
  const match = textoCard.match(/\$([0-9,.]+)/);
  if (match) {
    return parseInt(match[1].replace(/[,.]/g, ''));
  }
  return 0;
}

function extraerAno(motoCard) {
  const textoCard = motoCard.textContent;
  const match = textoCard.match(/Año:\s*(\d{4})/);
  if (match) {
    return parseInt(match[1]);
  }
  return 0;
}

function actualizarContadorResultados(cantidad) {
  const contador = document.getElementById('contadorResultados');
  if (contador) {
    contador.innerHTML = `<span class="font-semibold">${cantidad}</span> motocicleta${cantidad !== 1 ? 's' : ''} encontrada${cantidad !== 1 ? 's' : ''}`;
  }
}

function limpiarFiltros() {
  // Resetear filtros
  filtrosActivos = {
    condicion: 'todas',
    estado: 'todas',
    marca: '',
    precio: '',
    ano: '',
    busqueda: ''
  };
  
  // Limpiar campos de formulario
  document.getElementById('filtroMarca').value = '';
  document.getElementById('filtroPrecio').value = '';
  document.getElementById('filtroAno').value = '';
  document.getElementById('buscarMoto').value = '';
  
  // Actualizar botones
  document.querySelectorAll('.filtro-btn').forEach(btn => {
    btn.classList.remove('filtro-activo');
  });
  
  // Marcar "todas" como activo para condición y estado
  const btnTodasCondicion = document.querySelector('.filtro-btn[onclick*="condicion"][onclick*="todas"]');
  const btnTodasEstado = document.querySelector('.filtro-btn[onclick*="estado"][onclick*="todas"]');
  
  if (btnTodasCondicion) btnTodasCondicion.classList.add('filtro-activo');
  if (btnTodasEstado) btnTodasEstado.classList.add('filtro-activo');
  
  // Mostrar todas las motos
  filtrarMotosConFiltros();
}

function buscarMotosEnTiempoReal(termino) {
  filtrosActivos.busqueda = termino;
  filtrarMotosConFiltros();
}

function cargarMarcasEnFiltro() {
  const marcas = new Set();
  
  // Extraer marcas únicas de las motos disponibles
  motosDisponibles.forEach(moto => {
    if (moto.marca) {
      marcas.add(moto.marca);
    }
  });
  
  const selectMarca = document.getElementById('filtroMarca');
  if (selectMarca) {
    // Limpiar opciones existentes (excepto "Todas las marcas")
    selectMarca.innerHTML = '<option value="">Todas las marcas</option>';
    
    // Agregar marcas encontradas
    Array.from(marcas).sort().forEach(marca => {
      const option = document.createElement('option');
      option.value = marca;
      option.textContent = marca;
      selectMarca.appendChild(option);
    });
  }
}

// Función heredada del sistema anterior (compatibilidad)
function filtrarMotos(tipo) {
  if (tipo === 'todas') {
    aplicarFiltro('estado', 'todas');
  } else {
    aplicarFiltro('estado', tipo);
  }
}

// ====================================================================
// GESTIÓN DE MOTOS - ACTUALIZADA
// ====================================================================

async function cargarMotosDisponibles() {
  try {
    console.log('🔄 Cargando motos disponibles...');

    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const id_empresa = localStorage.getItem('id_empresa') || 1;

    if (!token) {
      console.warn('⚠️ Token no encontrado, trabajando en modo offline');
      mostrarMotosDemo();
      return;
    }

    const response = await fetch(`http://localhost:3001/api/motos/disponibles?id_empresa=${id_empresa}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });

    if (!response.ok) {
      console.warn('⚠️ Error del servidor al cargar motos, mostrando datos demo');
      mostrarMotosDemo();
      return;
    }

    const motos = await response.json();
    console.log('✅ Respuesta del servidor motos:', motos);

    if (motos && Array.isArray(motos) && motos.length > 0) {
      motosDisponibles = motos;
      mostrarMotosEnLista(motos);
      console.log(`✅ ${motos.length} motos cargadas`);
    } else {
      console.warn('⚠️ Respuesta del servidor sin motos, mostrando demo');
      mostrarMotosDemo();
    }

  } catch (error) {
    console.error('Error cargando motos:', error);
    console.log('🔄 Continuando en modo offline...');
    mostrarMotosDemo();
  }
}

// Función para mostrar motos demo en caso de error
function mostrarMotosDemo() {
  const motosDemo = [
    {
      id_moto: 1,
      marca: 'Honda',
      modelo: 'CB650R',
      año: 2024,
      anio: 2024,
      color: 'Negro Mate',
      patente: 'ABC-123',
      kms: 0,
      kilometraje: 0,
      soc: 'ME001',
      llaves: 2,
      precio: 9500000,
      precio_venta: 9500000,
      estado: 'DISPONIBLE',
      condicion: 'NUEVA',
      categoria: 'NAKED'
    },
    {
      id_moto: 2,
      marca: 'Yamaha',
      modelo: 'MT-07',
      año: 2023,
      anio: 2023,
      color: 'Azul',
      patente: 'XYZ-789',
      kms: 3500,
      kilometraje: 3500,
      soc: 'ME002',
      llaves: 2,
      precio: 8500000,
      precio_venta: 8500000,
      estado: 'DISPONIBLE',
      condicion: 'USADA',
      categoria: 'NAKED'
    },
    {
      id_moto: 3,
      marca: 'KTM',
      modelo: 'Duke 390',
      año: 2024,
      anio: 2024,
      color: 'Naranja',
      patente: 'KTM-456',
      kms: 500,
      kilometraje: 500,
      soc: 'ME003',
      llaves: 3,
      precio: 8900000,
      precio_venta: 8900000,
      estado: 'DISPONIBLE',
      condicion: 'NUEVA',
      categoria: 'NAKED'
    },
    {
      id_moto: 4,
      marca: 'BMW',
      modelo: 'R1250GS',
      año: 2022,
      anio: 2022,
      color: 'Gris',
      patente: 'BMW-321',
      kms: 12000,
      kilometraje: 12000,
      soc: 'ME004',
      llaves: 2,
      precio: 15500000,
      precio_venta: 15500000,
      estado: 'DISPONIBLE',
      condicion: 'USADA',
      categoria: 'ADVENTURE'
    }
  ];
  motosDisponibles = motosDemo;
  mostrarMotosEnLista(motosDemo);
}

function mostrarMotosEnLista(motos) {
  const container = document.getElementById('listaMotosVenta');
  
  if (!container) {
    console.error('Contenedor listaMotosVenta no encontrado');
    return;
  }
  
  if (!motos || motos.length === 0) {
    container.innerHTML = `
      <div class="text-center py-8 text-gray-500 dark:text-gray-400">
        <div class="text-4xl mb-2">🏍️</div>
        <p>No hay motos disponibles para la venta</p>
      </div>
    `;
    actualizarContadorResultados(0);
    return;
  }
  
  container.innerHTML = motos.map(moto => `
    <div class="moto-card bg-white dark:bg-gray-700 border border-gray-200 dark:border-gray-600 rounded-lg p-4 hover:shadow-md transition-all cursor-pointer" 
         onclick="seleccionarMoto(${JSON.stringify(moto).replace(/"/g, '&quot;')}, this)"
         data-marca="${moto.marca || ''}"
         data-modelo="${moto.modelo || ''}"
         data-estado="${moto.estado || ''}"
         data-condicion="${moto.condicion || ''}"
         data-patente="${moto.patente || ''}"
         data-precio="${moto.precio_venta || moto.precio || 0}"
         data-ano="${moto.ano || moto.año || 0}">
      <div class="flex justify-between items-start mb-3">
        <div>
          <h4 class="font-semibold text-gray-900 dark:text-white">${moto.marca} ${moto.modelo}</h4>
          <p class="text-sm text-gray-600 dark:text-gray-400">Año: ${moto.ano || moto.año || 'N/A'}</p>
          ${moto.patente ? `<p class="text-xs text-blue-600 dark:text-blue-400 font-mono">📄 ${moto.patente}</p>` : ''}
        </div>
        <div class="text-right">
          <div class="text-lg font-bold text-green-600 dark:text-green-400">
            $${(moto.precio_venta || moto.precio || 0).toLocaleString('es-CL')}
          </div>
          <div class="text-xs text-gray-500">Precio venta</div>
        </div>
      </div>
      
      <div class="grid grid-cols-2 gap-2 text-sm">
        <div>
          <span class="text-gray-500">Color:</span>
          <span class="font-medium text-gray-900 dark:text-white">${moto.color || 'N/A'}</span>
        </div>
        <div>
          <span class="text-gray-500">Kilómetros:</span>
          <span class="font-medium text-gray-900 dark:text-white">${(moto.kilometraje || moto.kms || 0).toLocaleString('es-CL')}</span>
        </div>
        <div>
          <span class="text-gray-500">SOC:</span>
          <span class="font-medium text-gray-900 dark:text-white">${moto.soc || 'N/A'}</span>
        </div>
        <div>
          <span class="text-gray-500">Llaves:</span>
          <span class="font-medium text-gray-900 dark:text-white">${moto.llaves || 'N/A'}</span>
        </div>
        <div>
          <span class="text-gray-500">Categoría:</span>
          <span class="font-medium text-gray-900 dark:text-white">${moto.categoria || 'N/A'}</span>
        </div>
        <div>
          <span class="text-gray-500">Estado:</span>
          <span class="inline-block px-2 py-1 text-xs rounded-full ${
            moto.estado === 'DISPONIBLE' ? 'bg-green-100 text-green-800' : 
            moto.estado === 'RESERVADA' ? 'bg-yellow-100 text-yellow-800' : 
            'bg-gray-100 text-gray-800'
          }">${moto.estado || 'N/A'}</span>
        </div>
      </div>
      
      ${moto.observaciones ? `
        <div class="mt-2 pt-2 border-t border-gray-200 dark:border-gray-600">
          <p class="text-xs text-gray-500">${moto.observaciones}</p>
        </div>
      ` : ''}
    </div>
  `).join('');
  
  // Inicializar filtros después de cargar las motos
  setTimeout(() => {
    cargarMarcasEnFiltro();
    actualizarContadorResultados(motos.length);
    
    // Marcar filtros por defecto como activos
    const btnTodasEstado = document.querySelector('.filtro-btn[onclick*="estado"][onclick*="todas"]');
    const btnTodasCondicion = document.querySelector('.filtro-btn[onclick*="condicion"][onclick*="todas"]');
    
    if (btnTodasEstado) btnTodasEstado.classList.add('filtro-activo');
    if (btnTodasCondicion) btnTodasCondicion.classList.add('filtro-activo');
  }, 100);
}

function seleccionarMoto(moto, elemento) {
  motoVenta = moto;
  console.log('🏍️ Moto seleccionada:', moto);
  
  // Remover selección previa
  document.querySelectorAll('#listaMotosVenta .moto-card').forEach(el => {
    el.classList.remove('selected', 'border-green-500', 'bg-green-50', 'dark:bg-green-900/20');
  });
  
  // Marcar como seleccionada - usar el elemento si se pasa, sino buscar el target del evento
  const targetElement = elemento || event?.currentTarget;
  if (targetElement) {
    targetElement.classList.add('selected', 'border-green-500', 'bg-green-50', 'dark:bg-green-900/20');
  }
  
  // Habilitar botón de continuar
  const btnContinuar = document.getElementById('btn-continuar-paso2');
  if (btnContinuar) {
    btnContinuar.disabled = false;
  }
  
  // Pre-llenar precio
  const precioInput = document.getElementById('precioVenta');
  if (precioInput && (moto.precio_venta || moto.precio)) {
    precioInput.value = moto.precio_venta || moto.precio;
  }
}

function mostrarDetallesVenta() {
  const detallesDiv = document.getElementById('detallesVenta');
  if (!detallesDiv) {
    console.error('Elemento detallesVenta no encontrado');
    return;
  }
  
  detallesDiv.classList.remove('hidden');
  
  // Pre-llenar precio de venta con el precio de la moto
  const precioInput = document.getElementById('precioVenta');
  if (precioInput && motoVenta) {
    precioInput.value = motoVenta.precio_venta || motoVenta.precio || '';
  }
  
  // Scroll hasta los detalles
  detallesDiv.scrollIntoView({ behavior: 'smooth', block: 'center' });
}

function limpiarFormularioVenta() {
  // Limpiar selecciones
  clienteVenta = null;
  motoVenta = null;
  
  // Limpiar campos - con verificación de existencia
  const buscarCliente = document.getElementById('buscarCliente');
  if (buscarCliente) buscarCliente.value = '';
  
  const precioVenta = document.getElementById('precioVenta');
  if (precioVenta) precioVenta.value = '';
  
  const descuentoVenta = document.getElementById('descuentoVenta');
  if (descuentoVenta) descuentoVenta.value = '';
  
  const metodoPago = document.getElementById('metodoPago');
  if (metodoPago) metodoPago.value = 'EFECTIVO';
  
  const observacionesVenta = document.getElementById('observacionesVenta');
  if (observacionesVenta) observacionesVenta.value = '';
  
  // Ocultar elementos - con verificación de existencia
  const detallesVenta = document.getElementById('detallesVenta');
  if (detallesVenta) detallesVenta.classList.add('hidden');
  
  const resultadosClientes = document.getElementById('resultadosClientes');
  if (resultadosClientes) resultadosClientes.classList.add('hidden');
  
  // Limpiar info cliente seleccionado
  const infoDiv = document.getElementById('infoclienteVenta');
  if (infoDiv) infoDiv.remove();
  
  // Remover selección visual de motos
  const listaMotos = document.getElementById('listaMotosVenta');
  if (listaMotos) {
    listaMotos.querySelectorAll('.bg-blue-50').forEach(el => {
      el.classList.remove('bg-blue-50', 'dark:bg-blue-900/20', 'border-blue-300', 'dark:border-blue-600');
    });
  }
}

// ====================================================================
// GESTIÓN DE CLIENTES - Sobrescribir funciones del global.js
// ====================================================================

// Sobrescribir la función mostrarmotoVenta del global.js
window.mostrarmotoVenta = function(moto) {
  console.log('Moto seleccionada en venta:', moto);
  // Esta función se maneja en seleccionarMoto específica de venta
};

// ====================================================================
// FUNCIONES DE CARGA DE DATOS
// ====================================================================

// Función para cargar información de la empresa
async function cargarInformacionEmpresa() {
  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const id_empresa = localStorage.getItem('id_empresa');
    
    if (!token || !id_empresa) {
      console.log('⚠️ No hay token o id_empresa, usando datos por defecto');
      empresaActual = {
        nombre: 'MotoExperts Santiago',
        rut: '76.123.456-7',
        direccion: 'Av. Providencia 1234, Santiago',
        telefono: '+56 9 1234 5678',
        email: 'contacto@motoexperts.cl'
      };
      return;
    }
    
    const response = await fetch(`http://localhost:3001/api/empresa/${id_empresa}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });
    
    if (response.ok) {
      const data = await response.json();
      empresaActual = data.data || data;
      console.log('✅ Información de empresa cargada:', empresaActual.nombre);
    } else {
      throw new Error(`Error ${response.status}: ${response.statusText}`);
    }
  } catch (error) {
    console.error('❌ Error cargando información de empresa:', error);
    // Usar datos por defecto
    empresaActual = {
      nombre: 'MotoExperts Santiago',
      rut: '76.123.456-7',
      direccion: 'Av. Providencia 1234, Santiago',
      telefono: '+56 9 1234 5678',
      email: 'contacto@motoexperts.cl'
    };
  }
}

async function cargarInformacionVendedor() {
  try {
    const datosGuardados = localStorage.getItem('datosUsuario');
    
    if (!datosGuardados) {
      console.log('⚠️ No hay datos de usuario, usando vendedor por defecto');
      vendedorActual = {
        nombre: 'Vendedor Sistema',
        email: 'vendedor@sistema.cl',
        id: 1
      };
      return;
    }
    
    const datosUsuario = JSON.parse(datosGuardados);
    vendedorActual = {
      nombre: datosUsuario.usuario?.nombre || 'Vendedor Sistema',
      email: datosUsuario.usuario?.correo || 'vendedor@sistema.cl',
      id: datosUsuario.usuario?.id || 1
    };
    
    console.log('✅ Información de vendedor cargada:', vendedorActual.nombre);
  } catch (error) {
    console.error('❌ Error cargando información de vendedor:', error);
    // Usar datos por defecto
    vendedorActual = {
      nombre: 'Vendedor Sistema',
      email: 'vendedor@sistema.cl',
      id: 1
    };
  }
}

// ====================================================================
// GESTIÓN DE CLIENTES Y REGIONES
// ====================================================================

// Función para cargar clientes
async function cargarClientes() {
  try {
    console.log('🔄 Cargando clientes...');
    
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const id_empresa = localStorage.getItem('id_empresa') || 1;
    
    if (!token) {
      console.warn('⚠️ Token no encontrado, trabajando en modo offline');
      mostrarClientesDemo();
      return;
    }
    
    const response = await fetch(`http://localhost:3001/api/clientes?empresa=${id_empresa}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    });
    
    if (!response.ok) {
      console.warn('⚠️ Error del servidor al cargar clientes, continuando en modo offline');
      mostrarClientesDemo();
      return;
    }
    
    const data = await response.json();
    console.log('✅ Respuesta del servidor:', data);
    
    if (data.success && data.data) {
      mostrarClientesEnLista(data.data);
    } else {
      mostrarClientesDemo();
    }
    
  } catch (error) {
    console.error('❌ Error al cargar clientes:', error);
    mostrarClientesDemo();
  }
}

// Función para mostrar clientes demo en caso de error
function mostrarClientesDemo() {
  const clientesDemo = [
    {
      id: 1,
      nombres: 'Carlos',
      apellidos: 'González',
      cedula: '12.345.678-9',
      telefono: '+56912345678',
      email: 'carlos@email.com'
    },
    {
      id: 2,
      nombres: 'Andrea',
      apellidos: 'Muñoz',
      cedula: '13.456.789-0',
      telefono: '+56923456789',
      email: 'andrea@email.com'
    }
  ];
  mostrarClientesEnLista(clientesDemo);
}

// Función para mostrar clientes en la lista
function mostrarClientesEnLista(clientes) {
  const contenedor = document.getElementById('listaClientes');
  if (!contenedor) return;

  if (!clientes || clientes.length === 0) {
    contenedor.innerHTML = `
      <div class="text-center py-8 text-gray-500 dark:text-gray-400">
        <svg class="w-12 h-12 mx-auto mb-4 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"></path>
        </svg>
        <p>No hay clientes registrados</p>
      </div>
    `;
    return;
  }

  const html = clientes.map(cliente => `
    <div class="border border-gray-200 dark:border-gray-700 rounded-xl p-4 hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-all cursor-pointer group" 
         onclick="seleccionarCliente(${JSON.stringify(cliente).replace(/"/g, '&quot;')})">
      <div class="flex items-center justify-between">
        <div class="flex items-center space-x-3">
          <div class="w-10 h-10 bg-blue-500 rounded-full flex items-center justify-center text-white font-semibold">
            ${cliente.nombres ? cliente.nombres.charAt(0).toUpperCase() : 'C'}
          </div>
          <div>
            <h5 class="font-semibold text-gray-900 dark:text-white group-hover:text-blue-600 dark:group-hover:text-blue-400 transition-colors">
              ${cliente.nombres || ''} ${cliente.apellidos || ''}
            </h5>
            <p class="text-sm text-gray-600 dark:text-gray-400">
              RUT: ${cliente.cedula || 'No especificado'}
            </p>
          </div>
        </div>
        <div class="text-right">
          <p class="text-sm text-gray-500 dark:text-gray-400">
            ${cliente.telefono || ''}
          </p>
          <p class="text-xs text-gray-400 dark:text-gray-500">
            ${cliente.email || ''}
          </p>
        </div>
      </div>
    </div>
  `).join('');

  contenedor.innerHTML = html;
}

// Función para buscar clientes
function buscarClientes() {
  const termino = document.getElementById('buscarCliente')?.value?.toLowerCase() || '';
  console.log('🔍 Buscando clientes:', termino);
  
  // Implementar filtrado local o llamada al servidor
  cargarClientes();
}

// Función para cargar regiones de Chile
async function cargarRegiones() {
  try {
    const regionSelect = document.getElementById('clienteRegion');
    if (!regionSelect) return;

    // Regiones de Chile
    const regiones = [
      'Región de Arica y Parinacota',
      'Región de Tarapacá',
      'Región de Antofagasta',
      'Región de Atacama',
      'Región de Coquimbo',
      'Región de Valparaíso',
      'Región Metropolitana de Santiago',
      'Región del Libertador General Bernardo O\'Higgins',
      'Región del Maule',
      'Región de Ñuble',
      'Región del Biobío',
      'Región de La Araucanía',
      'Región de Los Ríos',
      'Región de Los Lagos',
      'Región Aysén del General Carlos Ibáñez del Campo',
      'Región de Magallanes y de la Antártica Chilena'
    ];

    regionSelect.innerHTML = '<option value="">Seleccionar región</option>';
    
    regiones.forEach(region => {
      const option = document.createElement('option');
      option.value = region;
      option.textContent = region;
      regionSelect.appendChild(option);
    });

  } catch (error) {
    console.error('❌ Error al cargar regiones:', error);
  }
}

// ====================================================================
// FUNCIONES AUXILIARES Y FORMULARIOS
// ====================================================================

// Función para toggle del formulario de cliente
function toggleFormularioCliente() {
  const formulario = document.getElementById('formularioCliente');
  if (formulario) {
    formulario.classList.toggle('hidden');
  }
}

// Función para crear nuevo cliente
async function crearNuevoCliente() {
  try {
    const nombres = document.getElementById('clienteNombres')?.value.trim();
    const apellidos = document.getElementById('clienteApellidos')?.value.trim();
    const cedula = document.getElementById('clienteCedula')?.value.trim();
    const tipo = document.getElementById('clienteTipo')?.value;
    const telefono = document.getElementById('clienteTelefono')?.value.trim();
    const email = document.getElementById('clienteEmail')?.value.trim();
    const region = document.getElementById('clienteRegion')?.value;
    const direccion = document.getElementById('clienteDireccion')?.value.trim();

    if (!nombres || !apellidos || !cedula) {
      Swal.fire({
        icon: 'warning',
        title: 'Campos requeridos',
        text: 'Por favor complete los campos obligatorios: Nombres, Apellidos y RUT',
        confirmButtonColor: '#3b82f6'
      });
      return;
    }

    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    const id_empresa = localStorage.getItem('id_empresa');

    const clienteData = {
      nombres,
      apellidos,
      cedula,
      tipo: tipo || 'NATURAL',
      telefono,
      email,
      region,
      direccion,
      id_empresa
    };

    const response = await fetch('http://localhost:3001/api/clientes', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(clienteData)
    });

    if (!response.ok) {
      throw new Error(`Error ${response.status}: ${response.statusText}`);
    }

    const nuevoCliente = await response.json();

    Swal.fire({
      icon: 'success',
      title: 'Cliente creado',
      text: 'El cliente ha sido creado exitosamente',
      confirmButtonColor: '#10b981'
    });

    // Limpiar formulario
    document.getElementById('clienteNombres').value = '';
    document.getElementById('clienteApellidos').value = '';
    document.getElementById('clienteCedula').value = '';
    document.getElementById('clienteTipo').value = '';
    document.getElementById('clienteTelefono').value = '';
    document.getElementById('clienteEmail').value = '';
    document.getElementById('clienteRegion').value = '';
    document.getElementById('clienteDireccion').value = '';

    // Ocultar formulario
    document.getElementById('formularioCliente').classList.add('hidden');

    // Recargar lista de clientes
    await cargarClientes();

    // Seleccionar automáticamente el cliente recién creado
    clienteVenta = nuevoCliente;
    actualizarVistaclienteVenta();

  } catch (error) {
    console.error('❌ Error al crear cliente:', error);
    Swal.fire({
      icon: 'error',
      title: 'Error',
      text: 'Error al crear el cliente: ' + error.message,
      confirmButtonColor: '#ef4444'
    });
  }
}

// Función para actualizar vista del cliente seleccionado
function actualizarVistaclienteVenta() {
  if (clienteVenta) {
    const infoCliente = document.getElementById('infoclienteVenta');
    if (infoCliente) {
      infoCliente.innerHTML = `
        <div class="flex items-center justify-between p-4 bg-green-50 dark:bg-green-900/20 rounded-xl border border-green-200 dark:border-green-700">
          <div class="flex items-center space-x-3">
            <div class="bg-green-500 text-white rounded-full w-10 h-10 flex items-center justify-center font-bold">
              ✓
            </div>
            <div>
              <h4 class="font-semibold text-gray-900 dark:text-white">${clienteVenta.nombres} ${clienteVenta.apellidos}</h4>
              <p class="text-sm text-gray-600 dark:text-gray-400">RUT: ${clienteVenta.cedula}</p>
            </div>
          </div>
          <button onclick="cambiarCliente()" class="text-blue-500 hover:text-blue-700 transition-colors">Cambiar</button>
        </div>
      `;
      infoCliente.classList.remove('hidden');
    }
  }
}

// Función para cambiar cliente
function cambiarCliente() {
  clienteVenta = null;
  const infoCliente = document.getElementById('infoclienteVenta');
  if (infoCliente) {
    infoCliente.classList.add('hidden');
  }
}

// Función para procesar la venta
async function procesarVenta() {
  console.log('🔄 Iniciando procesamiento de venta, tipo:', tipoOperacion);
  
  // Validar que tenemos toda la información necesaria
  if (!clienteVenta) {
    Swal.fire({
      icon: 'warning',
      title: 'Cliente requerido',
      text: 'Por favor seleccione un cliente',
      confirmButtonColor: '#3b82f6'
    });
    return;
  }

  if (!motoVenta) {
    Swal.fire({
      icon: 'warning',
      title: 'Moto requerida',
      text: 'Por favor seleccione una moto',
      confirmButtonColor: '#3b82f6'
    });
    return;
  }

  if (!tipoOperacion) {
    Swal.fire({
      icon: 'warning',
      title: 'Tipo de operación requerido',
      text: 'Por favor seleccione el tipo de operación en el paso anterior',
      confirmButtonColor: '#3b82f6'
    });
    return;
  }

  const precioVenta = document.getElementById('precioVenta')?.value;
  const metodoPago = document.getElementById('metodoPago')?.value;

  if (!precioVenta || !metodoPago) {
    Swal.fire({
      icon: 'warning',
      title: 'Campos requeridos',
      text: 'Por favor complete el precio de venta y método de pago',
      confirmButtonColor: '#3b82f6'
    });
    return;
  }

  // Enrutar según el tipo de operación
  try {
    if (tipoOperacion === 'VENTA_DIRECTA') {
      // Para venta directa, validar que se haya completado la autenticación biométrica
      if (!huellaAutenticada) {
        Swal.fire({
          icon: 'warning',
          title: 'Autenticación requerida',
          text: 'Debe completar la validación biométrica para procesar una venta directa',
          confirmButtonColor: '#3b82f6'
        });
        return;
      }
      
      console.log('📋 Procesando VENTA DIRECTA con validación biométrica');
      await procesarVentaDirecta();
      
    } else if (tipoOperacion === 'RESERVAR') {
      console.log('🎫 Procesando RESERVA sin validación biométrica');
      await procesarReserva();
      
    } else {
      throw new Error('Tipo de operación no válido: ' + tipoOperacion);
    }
    
  } catch (error) {
    console.error('❌ Error en procesamiento:', error);
    Swal.fire({
      icon: 'error',
      title: 'Error en procesamiento',
      text: error.message || 'Ocurrió un error inesperado',
      confirmButtonColor: '#ef4444'
    });
  }
}

// Función para reiniciar el formulario
function reiniciarFormulario() {
  clienteVenta = null;
  motoVenta = null;
  tipoOperacion = null; // Resetear tipo de operación
  huellaAutenticada = false; // Resetear autenticación
  
  // Limpiar campos
  document.getElementById('precioVenta').value = '';
  document.getElementById('descuentoVenta').value = '';
  document.getElementById('metodoPago').value = 'EFECTIVO';
  document.getElementById('observacionesVenta').value = '';
  
  // Resetear selección de tipo de operación
  document.getElementById('opcion-reservar')?.classList.remove('border-orange-400', 'bg-orange-50');
  document.getElementById('opcion-reservar')?.classList.add('border-orange-200');
  document.getElementById('dot-reservar')?.classList.add('hidden');
  
  document.getElementById('opcion-venta-directa')?.classList.remove('border-green-400', 'bg-green-50');
  document.getElementById('opcion-venta-directa')?.classList.add('border-green-200');
  document.getElementById('dot-venta-directa')?.classList.add('hidden');
  
  // Ocultar alerta de selección
  document.getElementById('alerta-seleccion')?.classList.add('hidden');
  
  // Resetear texto del botón
  const textoContinuar = document.getElementById('texto-continuar');
  if (textoContinuar) {
    textoContinuar.textContent = 'Continuar';
  }
  
  // Ocultar secciones
  document.getElementById('infoclienteVenta')?.classList.add('hidden');
  document.getElementById('infomotoVenta')?.classList.add('hidden');
  document.getElementById('detallesVenta')?.classList.add('hidden');
  
  // Volver al primer paso
  pasoActual = 1;
  continuarAPaso(1);
}

// Función para reiniciar el proceso completo
function reiniciarProceso() {
  console.log('🔄 Reiniciando proceso de venta - Recargando página');
  
  // Simplemente recargar la página para empezar desde cero
  window.location.reload();
}

// Funciones auxiliares para PDFs
function descargarPDF(doc, nombreArchivo) {
  try {
    doc.save(`${nombreArchivo}.pdf`);
    console.log('📥 PDF descargado:', nombreArchivo);
  } catch (error) {
    console.error('❌ Error al descargar PDF:', error);
    mostrarError('Error', 'No se pudo descargar el archivo PDF');
  }
}

function imprimirPDF(pdfUrl) {
  try {
    const ventanaImpresion = window.open(pdfUrl, '_blank');
    if (ventanaImpresion) {
      ventanaImpresion.onload = function() {
        ventanaImpresion.print();
      };
    } else {
      throw new Error('No se pudo abrir la ventana de impresión');
    }
    console.log('🖨️ Enviando a impresión...');
  } catch (error) {
    console.error('❌ Error al imprimir PDF:', error);
    mostrarError('Error', 'No se pudo enviar el documento a imprimir');
  }
}

// Función debounce para búsquedas (si no está en global.js)
if (typeof window.debounce !== 'function') {
  window.debounce = function(func, wait) {
    let timeout;
    return function executedFunction(...args) {
      const later = () => {
        clearTimeout(timeout);
        func(...args);
      };
      clearTimeout(timeout);
      timeout = setTimeout(later, wait);
    };
  };
}

// ====================================================================
// FUNCIONES DE FORMATEO DE PRECIOS Y PAGO MIXTO
// ====================================================================

// Función para formatear precios con separadores de miles
function formatearPrecio(input) {
  // Obtener el valor actual sin formato
  let valor = input.value.replace(/[^\d]/g, '');
  
  // Si no hay valor, salir
  if (!valor) {
    input.value = '';
    return;
  }
  
  // Convertir a número y formatear con separadores de miles
  let numero = parseInt(valor);
  let valorFormateado = numero.toLocaleString('es-CL');
  
  // Asignar el valor formateado
  input.value = valorFormateado;
}

// Función para obtener el valor numérico de un campo formateado
function obtenerValorNumerico(inputId) {
  const input = document.getElementById(inputId);
  if (!input || !input.value) return 0;
  
  // Remover formato y convertir a número
  return parseInt(input.value.replace(/[^\d]/g, '')) || 0;
}

// Función para manejar cambio de método de pago
function manejarCambioMetodoPago() {
  const metodoPago = document.getElementById('metodoPago').value;
  const seccionMixto = document.getElementById('seccionPagoMixto');
  const seccionCuotas = document.getElementById('seccionCuotasCredito');
  
  // Ocultar todas las secciones primero
  seccionMixto.classList.add('hidden');
  seccionCuotas.classList.add('hidden');
  
  if (metodoPago === 'MIXTO') {
    seccionMixto.classList.remove('hidden');
    // Inicializar con un método de pago por defecto
    if (document.getElementById('metodosPagoMixto').children.length === 0) {
      agregarMetodoPagoMixto();
    }
  } else if (metodoPago === 'CREDITO') {
    seccionCuotas.classList.remove('hidden');
    // Calcular cuotas inicial
    calcularCuotas();
  } else {
    // Limpiar métodos mixtos si se cambia a otro método
    document.getElementById('metodosPagoMixto').innerHTML = '';
    document.getElementById('metodosPagoMixtoData').value = '';
  }
}

// Variable para contar métodos de pago mixto
let contadorMetodosMixto = 0;

// Función para agregar un método de pago mixto
function agregarMetodoPagoMixto() {
  contadorMetodosMixto++;
  const container = document.getElementById('metodosPagoMixto');
  
  const metodoDiv = document.createElement('div');
  metodoDiv.className = 'flex items-center space-x-3 bg-white dark:bg-gray-600 p-3 rounded border';
  metodoDiv.id = `metodo-mixto-${contadorMetodosMixto}`;
  
  metodoDiv.innerHTML = `
    <select class="flex-1 px-3 py-2 border border-gray-300 dark:border-gray-500 rounded text-sm dark:bg-gray-700 dark:text-white" onchange="actualizarMetodosMixtos()">
      <option value="">Seleccionar método</option>
      <option value="EFECTIVO">💵 Efectivo</option>
      <option value="TRANSFERENCIA">🏦 Transferencia</option>
      <option value="DEBITO">💳 Débito</option>
      <option value="CREDITO">💳 Crédito</option>
      <option value="VALE_VISTA">📄 Vale Vista</option>
    </select>
    <div class="relative flex-1">
      <div class="absolute inset-y-0 left-0 pl-2 flex items-center pointer-events-none">
        <span class="text-gray-500 text-sm">$</span>
      </div>
      <input type="text" placeholder="Monto" class="w-full pl-6 pr-3 py-2 border border-gray-300 dark:border-gray-500 rounded text-sm dark:bg-gray-700 dark:text-white" oninput="formatearPrecio(this); actualizarMetodosMixtos()">
    </div>
    <button type="button" onclick="eliminarMetodoPagoMixto('metodo-mixto-${contadorMetodosMixto}')" class="text-red-500 hover:text-red-700 p-1">
      <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
      </svg>
    </button>
  `;
  
  container.appendChild(metodoDiv);
  actualizarMetodosMixtos();
}

// Función para eliminar un método de pago mixto
function eliminarMetodoPagoMixto(metodoId) {
  const metodo = document.getElementById(metodoId);
  if (metodo) {
    metodo.remove();
    actualizarMetodosMixtos();
  }
}

// Función para actualizar la data de métodos mixtos
function actualizarMetodosMixtos() {
  const metodos = [];
  const container = document.getElementById('metodosPagoMixto');
  
  Array.from(container.children).forEach(metodoDiv => {
    const select = metodoDiv.querySelector('select');
    const input = metodoDiv.querySelector('input');
    
    if (select.value && input.value) {
      const monto = parseInt(input.value.replace(/[^\d]/g, '')) || 0;
      if (monto > 0) {
        metodos.push({
          metodo: select.value,
          monto: monto
        });
      }
    }
  });
  
  // Guardar en el campo oculto
  document.getElementById('metodosPagoMixtoData').value = JSON.stringify(metodos);
  
  // Actualizar resumen si es necesario
  actualizarResumen();
}

// Función actualizada para el resumen que considera formateo
function actualizarResumen() {
  // Obtener valores numéricos
  const precio = obtenerValorNumerico('precioVenta');
  const descuento = obtenerValorNumerico('descuentoVenta');
  const total = precio - descuento;
  
  // Actualizar campos de resumen con formato
  const resumenPrecio = document.getElementById('resumenPrecio');
  const resumenDescuento = document.getElementById('resumenDescuento');
  const resumenTotal = document.getElementById('resumenTotal');
  
  if (resumenPrecio) {
    resumenPrecio.textContent = `$${precio.toLocaleString('es-CL')}`;
  }
  
  if (resumenDescuento) {
    resumenDescuento.textContent = `-$${descuento.toLocaleString('es-CL')}`;
  }
  
  if (resumenTotal) {
    resumenTotal.textContent = `$${total.toLocaleString('es-CL')}`;
  }
  
  // Actualizar cuotas si está activo el pago con crédito
  const metodoPago = document.getElementById('metodoPago').value;
  if (metodoPago === 'CREDITO') {
    calcularCuotas();
  }
  
  // Validar pago mixto si está activo
  if (metodoPago === 'MIXTO') {
    validarPagoMixto(total);
  }
}

// Función para validar que el pago mixto cubra el total
function validarPagoMixto(totalVenta) {
  const metodosMixtoData = document.getElementById('metodosPagoMixtoData').value;
  if (!metodosMixtoData) return;
  
  try {
    const metodos = JSON.parse(metodosMixtoData);
    const totalMixto = metodos.reduce((sum, metodo) => sum + metodo.monto, 0);
    
    // Mostrar advertencia si no coincide el monto
    let advertenciaExistente = document.getElementById('advertencia-pago-mixto');
    
    if (totalMixto !== totalVenta) {
      if (!advertenciaExistente) {
        const advertencia = document.createElement('div');
        advertencia.id = 'advertencia-pago-mixto';
        advertencia.className = 'mt-2 p-2 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-700 rounded text-sm';
        advertencia.innerHTML = `
          <div class="flex items-center text-yellow-700 dark:text-yellow-300">
            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.664-.833-2.464 0L3.34 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
            </svg>
            <span>Total mixto: $${totalMixto.toLocaleString('es-CL')} | Falta: $${(totalVenta - totalMixto).toLocaleString('es-CL')}</span>
          </div>
        `;
        document.getElementById('seccionPagoMixto').appendChild(advertencia);
      } else {
        // Actualizar mensaje existente
        const span = advertenciaExistente.querySelector('span');
        span.textContent = `Total mixto: $${totalMixto.toLocaleString('es-CL')} | Falta: $${(totalVenta - totalMixto).toLocaleString('es-CL')}`;
      }
    } else {
      // Remover advertencia si el total es correcto
      if (advertenciaExistente) {
        advertenciaExistente.remove();
      }
    }
  } catch (error) {
    console.error('Error al validar pago mixto:', error);
  }
}

// ====================================================================
// CÁLCULO DE CUOTAS PARA TARJETA DE CRÉDITO
// ====================================================================

// Tasas de interés por número de cuotas (porcentaje mensual)
const tasasInteres = {
  1: 0,      // Sin interés
  3: 1.5,    // 1.5% mensual
  6: 1.8,    // 1.8% mensual  
  9: 2.0,    // 2.0% mensual
  12: 2.2,   // 2.2% mensual
  18: 2.5,   // 2.5% mensual
  24: 2.8,   // 2.8% mensual
  36: 3.0    // 3.0% mensual
};

function calcularCuotas() {
  const numeroCuotas = parseInt(document.getElementById('numeroCuotas').value) || 1;
  const precio = obtenerValorNumerico('precioVenta');
  const descuento = obtenerValorNumerico('descuentoVenta');
  const montoTotal = precio - descuento;
  
  if (montoTotal <= 0) {
    document.getElementById('valorCuota').value = '$0';
    document.getElementById('totalConCuotas').textContent = '$0';
    document.getElementById('textoInteres').textContent = '';
    return;
  }
  
  const tasaMensual = tasasInteres[numeroCuotas] || 0;
  
  if (numeroCuotas === 1 || tasaMensual === 0) {
    // Sin interés
    const valorCuota = montoTotal;
    document.getElementById('valorCuota').value = `$${valorCuota.toLocaleString('es-CL')}`;
    document.getElementById('totalConCuotas').textContent = `$${montoTotal.toLocaleString('es-CL')}`;
    document.getElementById('textoInteres').textContent = '(sin interés)';
  } else {
    // Con interés compuesto
    const tasaDecimal = tasaMensual / 100;
    const factor = Math.pow(1 + tasaDecimal, numeroCuotas);
    const valorCuota = Math.round((montoTotal * tasaDecimal * factor) / (factor - 1));
    const totalConInteres = valorCuota * numeroCuotas;
    const interesTotal = totalConInteres - montoTotal;
    
    document.getElementById('valorCuota').value = `$${valorCuota.toLocaleString('es-CL')}`;
    document.getElementById('totalConCuotas').textContent = `$${totalConInteres.toLocaleString('es-CL')}`;
    document.getElementById('textoInteres').textContent = `(+$${interesTotal.toLocaleString('es-CL')} interés)`;
  }
}

console.log('📦 Operaciones Venta.js cargado correctamente');

// ====================================================================
// FUNCIONES AUXILIARES GLOBALES
// ====================================================================

// Función para mostrar errores (si no está en global.js)
if (typeof window.mostrarError !== 'function') {
  window.mostrarError = function(titulo, mensaje) {
    Swal.fire({
      icon: 'error',
      title: titulo,
      text: mensaje,
      confirmButtonColor: '#ef4444'
    });
  };
}

// Función para mostrar loading (si no está en global.js)
if (typeof window.mostrarLoading !== 'function') {
  window.mostrarLoading = function(mensaje = 'Procesando...') {
    Swal.fire({
      title: mensaje,
      allowOutsideClick: false,
      showConfirmButton: false,
      willOpen: () => {
        Swal.showLoading();
      }
    });
  };
}

// Función para cerrar loading (si no está en global.js)
if (typeof window.cerrarLoading !== 'function') {
  window.cerrarLoading = function() {
    Swal.close();
  };
}

// ====================================================================
// FUNCIONES ADICIONALES PARA EL NUEVO FLUJO
// ====================================================================
// RESUMEN DE VENTA
// ====================================================================

function mostrarResumenVenta() {
  console.log('📋 Mostrando resumen de venta');
  
  // Mostrar información del cliente
  if (clienteVenta) {
    document.getElementById('resumenClienteInfo').innerHTML = `
      <div class="flex justify-between">
        <span class="text-gray-600 dark:text-gray-400">Nombre:</span>
        <span class="font-medium text-gray-900 dark:text-white">${clienteVenta.nombres} ${clienteVenta.apellidos}</span>
      </div>
      <div class="flex justify-between">
        <span class="text-gray-600 dark:text-gray-400">RUT:</span>
        <span class="font-medium text-gray-900 dark:text-white">${clienteVenta.cedula}</span>
      </div>
      ${clienteVenta.telefono ? `
        <div class="flex justify-between">
          <span class="text-gray-600 dark:text-gray-400">Teléfono:</span>
          <span class="font-medium text-gray-900 dark:text-white">${clienteVenta.telefono}</span>
        </div>
      ` : ''}
    `;
  }
  
  // Mostrar información de la moto
  if (motoVenta) {
    document.getElementById('resumenMotoInfo').innerHTML = `
      <div class="grid grid-cols-1 gap-2">
        <!-- Información básica -->
        <div class="flex justify-between border-b border-gray-100 dark:border-gray-600 pb-2">
          <span class="text-gray-600 dark:text-gray-400 font-medium">Marca/Modelo:</span>
          <span class="font-semibold text-gray-900 dark:text-white">${motoVenta.marca} ${motoVenta.modelo}</span>
        </div>
        
        <div class="flex justify-between">
          <span class="text-gray-600 dark:text-gray-400">Año:</span>
          <span class="font-medium text-gray-900 dark:text-white">${motoVenta.ano || motoVenta.año || 'N/A'}</span>
        </div>
        
        <div class="flex justify-between">
          <span class="text-gray-600 dark:text-gray-400">Color:</span>
          <span class="font-medium text-gray-900 dark:text-white">${motoVenta.color || 'N/A'}</span>
        </div>
        
        <div class="flex justify-between">
          <span class="text-gray-600 dark:text-gray-400">Patente:</span>
          <span class="font-medium font-mono text-blue-600 dark:text-blue-400">${motoVenta.patente || 'N/A'}</span>
        </div>
        
        <!-- Especificaciones técnicas -->
        ${motoVenta.categoria ? `
        <div class="flex justify-between">
          <span class="text-gray-600 dark:text-gray-400">Categoría:</span>
          <span class="font-medium">
            <span class="inline-block px-2 py-1 text-xs rounded-full bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200">
              ${motoVenta.categoria}
            </span>
          </span>
        </div>
        ` : ''}
        
        <div class="flex justify-between">
          <span class="text-gray-600 dark:text-gray-400">Kilómetros:</span>
          <span class="font-medium text-gray-900 dark:text-white">${(motoVenta.kms || motoVenta.kilometraje || 0).toLocaleString('es-CL')} km</span>
        </div>
        
        ${motoVenta.soc ? `
        <div class="flex justify-between">
          <span class="text-gray-600 dark:text-gray-400">SOC:</span>
          <span class="font-medium font-mono text-gray-900 dark:text-white">${motoVenta.soc}</span>
        </div>
        ` : ''}
        
        ${motoVenta.llaves ? `
        <div class="flex justify-between">
          <span class="text-gray-600 dark:text-gray-400">Llaves:</span>
          <span class="font-medium text-gray-900 dark:text-white">${motoVenta.llaves} ${motoVenta.llaves === 1 ? 'llave' : 'llaves'}</span>
        </div>
        ` : ''}
        
        <!-- Estado comercial -->
        <div class="flex justify-between border-t border-gray-100 dark:border-gray-600 pt-2 mt-2">
          <span class="text-gray-600 dark:text-gray-400">Estado:</span>
          <span class="font-medium text-gray-900 dark:text-white">
            <span class="inline-block px-2 py-1 text-xs rounded-full ${
              motoVenta.estado === 'DISPONIBLE' ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' : 
              motoVenta.estado === 'RESERVADA' ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200' : 
              'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-200'
            }">
              ${motoVenta.estado || 'N/A'}
            </span>
          </span>
        </div>
        
        <div class="flex justify-between">
          <span class="text-gray-600 dark:text-gray-400">Condición:</span>
          <span class="font-medium text-gray-900 dark:text-white">
            <span class="inline-block px-2 py-1 text-xs rounded-full ${
              motoVenta.condicion === 'NUEVA' ? 'bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200' : 
              motoVenta.condicion === 'USADA' ? 'bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200' : 
              'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-200'
            }">
              ${motoVenta.condicion || 'N/A'}
            </span>
          </span>
        </div>
        
        ${motoVenta.origen ? `
        <div class="flex justify-between">
          <span class="text-gray-600 dark:text-gray-400">Origen:</span>
          <span class="font-medium text-gray-900 dark:text-white">
            <span class="inline-block px-2 py-1 text-xs rounded-full ${
              motoVenta.origen === 'PROPIA' ? 'bg-indigo-100 text-indigo-800 dark:bg-indigo-900 dark:text-indigo-200' : 
              motoVenta.origen === 'CONSIGNACION' ? 'bg-teal-100 text-teal-800 dark:bg-teal-900 dark:text-teal-200' : 
              'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-200'
            }">
              ${motoVenta.origen}
            </span>
          </span>
        </div>
        ` : ''}
        
        <!-- Precio -->
        <div class="flex justify-between border-t border-gray-100 dark:border-gray-600 pt-2 mt-2">
          <span class="text-gray-600 dark:text-gray-400 font-medium">Precio Publicación:</span>
          <span class="font-bold text-lg text-green-600 dark:text-green-400">
            $${(motoVenta.precio_venta || motoVenta.precio || 0).toLocaleString('es-CL')}
          </span>
        </div>
        
        ${motoVenta.observaciones ? `
        <div class="mt-3 pt-2 border-t border-gray-100 dark:border-gray-600">
          <span class="text-gray-600 dark:text-gray-400 text-sm font-medium">Observaciones:</span>
          <p class="text-sm text-gray-700 dark:text-gray-300 mt-1 italic">${motoVenta.observaciones}</p>
        </div>
        ` : ''}
      </div>
    `;
  }
  
  // Actualizar resumen financiero
  actualizarResumen();
}

// ====================================================================
// AUTENTICACIÓN CON HUELLA
// ====================================================================

function prepararAutenticacion() {
  console.log('🔐 Preparando autenticación biométrica');
  huellaAutenticada = false;
  
  // Resetear estados del escáner
  const scanner = document.getElementById('fingerprintScanner');
  const status = document.getElementById('fingerprintStatus');
  const progress = document.getElementById('fingerprintProgress');
  const btnFinalizar = document.getElementById('btn-finalizar-venta');
  const btnSimular = document.getElementById('btnSimularHuella');
  
  scanner.className = 'fingerprint-scanner mx-auto mb-6';
  status.textContent = 'Esperando lectura de huella...';
  progress.textContent = 'Coloca tu dedo firmemente en el sensor';
  btnFinalizar.disabled = true;
  
  // Asegurar que el botón de simular esté visible y habilitado
  if (btnSimular) {
    btnSimular.style.display = 'inline-flex';
    btnSimular.disabled = false;
    btnSimular.textContent = 'Simular Lectura de Huella';
  }
}

function simularLecturaHuella() {
  console.log('👆 Simulando lectura de huella dactilar');
  
  const scanner = document.getElementById('fingerprintScanner');
  const status = document.getElementById('fingerprintStatus');
  const progress = document.getElementById('fingerprintProgress');
  const btnSimular = document.getElementById('btnSimularHuella');
  const btnFinalizar = document.getElementById('btn-finalizar-venta');
  
  // Deshabilitar botón de simulación
  btnSimular.disabled = true;
  btnSimular.textContent = 'Escaneando...';
  
  // Iniciar animación de escaneo
  scanner.classList.add('scanning');
  status.textContent = 'Escaneando huella dactilar...';
  progress.textContent = 'Mantén el dedo quieto en el sensor';
  
  // Simular proceso de escaneo (3 segundos)
  setTimeout(() => {
    // Simular éxito (90% de probabilidad)
    const exito = Math.random() > 0.1;
    
    if (exito) {
      // Éxito en la autenticación
      scanner.classList.remove('scanning');
      scanner.classList.add('success');
      status.textContent = '✅ Huella autenticada correctamente';
      progress.textContent = 'Identidad verificada - Generando documento...';
      
      huellaAutenticada = true;
      btnFinalizar.disabled = false;
      btnSimular.style.display = 'none';
      
      // Mostrar animación de éxito 
      Swal.fire({
        icon: 'success',
        title: 'Autenticación Exitosa',
        text: 'Huella verificada correctamente. Ahora puedes finalizar la venta.',
        timer: 2000,
        showConfirmButton: false,
        toast: true,
        position: 'top-end'
      });
      
      console.log('✅ Huella autenticada - Usuario puede proceder a finalizar venta');
      
    } else {
      // Error en la autenticación
      scanner.classList.remove('scanning');
      scanner.classList.add('error');
      status.textContent = '❌ Error en la lectura';
      progress.textContent = 'Huella no reconocida - Intenta nuevamente';
      
      // Resetear después de 2 segundos
      setTimeout(() => {
        scanner.classList.remove('error');
        status.textContent = 'Esperando lectura de huella...';
        progress.textContent = 'Coloca tu dedo firmemente en el sensor';
        btnSimular.disabled = false;
        btnSimular.textContent = 'Simular Lectura de Huella';
      }, 2000);
      
      Swal.fire({
        icon: 'error',
        title: 'Error de Autenticación',
        text: 'No se pudo verificar tu identidad. Intenta nuevamente.',
        timer: 3000,
        showConfirmButton: false,
        toast: true,
        position: 'top-end'
      });
    }
  }, 3000);
}

function finalizarVenta() {
  console.log('🎯 INICIO finalizarVenta() - tipoOperacion:', tipoOperacion);
  console.log('🎯 huellaAutenticada:', huellaAutenticada);
  
  if (!huellaAutenticada) {
    mostrarError('Autenticación requerida', 'Debes autenticarte con tu huella dactilar antes de finalizar la venta');
    return;
  }
  
  console.log('🎉 Finalizando venta directa con autenticación');
  
  // Verificar que el tipo de operación sea VENTA_DIRECTA
  if (tipoOperacion !== 'VENTA_DIRECTA') {
    console.error('❌ Error: finalizarVenta() solo debe llamarse para VENTA_DIRECTA');
    console.error('❌ Tipo actual:', tipoOperacion);
    mostrarError('Error de flujo', 'Esta función solo debe ejecutarse para ventas directas');
    return;
  }
  
  console.log('✅ Llamando a procesarVentaDirecta()...');
  // Llamar directamente a procesarVentaDirecta ya que tenemos la autenticación
  procesarVentaDirecta();
}

// ====================================================================
// GENERACIÓN DE PDF - NOTA DE VENTA
// ====================================================================

// FUNCIÓN OBSOLETA - Usar generarNotaVentaReserva() o generarContratoVentaDirecta()
function generarNotaVentaPDF() {
  console.error('❌ FUNCIÓN OBSOLETA: No debe usarse generarNotaVentaPDF()');
  console.error('❌ Usar generarNotaVentaReserva() o generarContratoVentaDirecta() según el tipo');
  mostrarError('Error de desarrollo', 'Función obsoleta llamada. Contactar soporte técnico.');
  return null;
}
    doc.line(20, 135, 20, 165); // Línea izquierda
    doc.line(190, 135, 190, 165); // Línea derecha
    doc.line(20, 145, 190, 145); // Línea separadora
    doc.line(20, 155, 190, 155); // Línea separadora
    doc.line(20, 165, 190, 165); // Línea inferior
    doc.line(140, 135, 140, 165); // Línea vertical central
    
    // Contenido de la tabla
    doc.setFont('helvetica', 'bold');
    doc.text('Concepto', 25, 142);
    doc.text('Monto', 155, 142);
    
    doc.setFont('helvetica', 'normal');
    doc.text('Precio de venta', 25, 152);
    doc.text(`$${precio.toLocaleString('es-CL')}`, 145, 152);
    
    if (descuento > 0) {
      doc.text('Descuento', 25, 162);
      doc.text(`-$${descuento.toLocaleString('es-CL')}`, 145, 162);
    }
    
    // Total
    doc.setFillColor(240, 240, 240);
    doc.rect(20, 165, 170, 10, 'F');
    doc.setFont('helvetica', 'bold');
    doc.text('TOTAL A PAGAR', 25, 172);
    doc.text(`$${total.toLocaleString('es-CL')}`, 145, 172);
    
    // Método de pago
    doc.setFont('helvetica', 'normal');
    doc.text(`Método de pago: ${metodoPago}`, 20, 185);
    
    // ====== TÉRMINOS Y CONDICIONES ======
    doc.setFontSize(12);
    doc.setFont('helvetica', 'bold');
    doc.text('TÉRMINOS Y CONDICIONES', 20, 200);
    
    doc.setFontSize(8);
    doc.setFont('helvetica', 'normal');
    doc.text('• Esta nota de venta constituye un comprobante de la transacción realizada.', 20, 210);
    doc.text('• El cliente declara haber recibido la motocicleta en perfectas condiciones.', 20, 216);
    doc.text('• La garantía y términos específicos se rigen por el contrato de compra-venta.', 20, 222);
    doc.text('• Documento generado electrónicamente con validación biométrica.', 20, 228);
    
    // ====== FIRMAS ======
    doc.setFontSize(10);
    doc.setFont('helvetica', 'normal');
    
    // Firma del vendedor
    doc.line(20, 250, 80, 250);
    doc.text('Firma del Vendedor', 30, 258);
    doc.text('MOTO TRAINER', 35, 265);
    
    // Firma del cliente
    doc.line(130, 250, 190, 250);
    doc.text('Firma del Cliente', 140, 258);
    doc.text(nombreCliente, 135, 265);
    
// ====================================================================
// FUNCIONES AUXILIARES Y FORMULARIOS
// ====================================================================

// Estructura para Nota de Venta (RESERVAR)
const estructuraNotaVenta = {
  tipo_documento: 'NOTA_VENTA',
  requiere_huella: false,
  estado_moto_resultado: 'RESERVADA',
  
  secciones: {
    encabezado: {
      titulo: 'NOTA DE VENTA',
      subtitulo: 'Documento de Reserva',
      numero_documento: true,
      fecha_emision: true,
      vigencia: '15 días hábiles'
    },
    
    empresa: {
      logo: true,
      nombre: true,
      rut: true,
      direccion: true,
      telefono: true,
      email: true
    },
    
    cliente: {
      nombres_completos: true,
      rut_cedula: true,
      telefono: true,
      email: true,
      direccion: true
    },
    
    vehiculo: {
      marca_modelo: true,
      año: true,
      color: true,
      cilindrada: true,
      numero_motor: true,
      numero_chasis: true,
      patente: true,
      kilometraje: true,
      estado: true
    },
    
    comercial: {
      precio_lista: true,
      descuento: true,
      precio_final: true,
      valor_reserva: true,
      saldo_pendiente: true,
      forma_pago_reserva: true
    },
    
    condiciones: {
      plazo_para_completar: true,
      condiciones_reserva: true,
      politica_devolucion: true,
      lugar_entrega: true
    },
    
    validacion: {
      firma_cliente: true,
      nombre_cliente: true,
      firma_vendedor: true,
      nombre_vendedor: true,
      fecha_firma: true,
      huella_digital: false
    }
  }
};

// Estructura para Contrato (VENTA DIRECTA)
const estructuraContrato = {
  tipo_documento: 'CONTRATO',
  requiere_huella: true,
  estado_moto_resultado: 'VENDIDA',
  
  secciones: {
    encabezado: {
      titulo: 'CONTRATO DE COMPRAVENTA',
      subtitulo: 'Vehículo Motorizado',
      numero_contrato: true,
      fecha_contrato: true,
      ciudad: true
    },
    
    partes_contratantes: {
      vendedor: {
        tipo: 'Empresa',
        razon_social: true,
        rut: true,
        representante_legal: true,
        domicilio: true,
        telefono: true
      },
      comprador: {
        tipo: 'Persona Natural/Jurídica',
        nombres_completos: true,
        rut: true,
        domicilio: true,
        telefono: true,
        email: true
      }
    },
    
    objeto_contrato: {
      descripcion_vehiculo: true,
      especificaciones_tecnicas: true,
      equipamiento_incluido: true,
      documentos_incluidos: true,
      estado_vehiculo: true,
      garantias_aplicables: true
    },
    
    clausulas_comerciales: {
      precio_total: true,
      forma_pago: true,
      descuentos_aplicados: true,
      impuestos: true,
      fecha_entrega: true,
      lugar_entrega: true
    },
    
    clausulas_legales: {
      transferencia_dominio: true,
      responsabilidad_vendedor: true,
      responsabilidad_comprador: true,
      garantia_legal: true,
      resolucion_conflictos: true,
      ley_aplicable: true
    },
    
    validacion_biometrica: {
      firma_vendedor: true,
      nombre_vendedor: true,
      cargo_vendedor: true,
      firma_comprador: true,
      nombre_comprador: true,
      huella_digital_comprador: true,
      fecha_firma: true,
      hora_firma: true,
      lugar_firma: true
    }
  }
};

// ====================================================================
// FUNCIONES DE GENERACIÓN DE DOCUMENTOS
// ====================================================================

// Función para generar Nota de Venta (RESERVAR)
function generarNotaVentaReserva() {
  try {
    console.log('📄 Generando Nota de Venta para RESERVA');
    
    if (typeof window.jspdf === 'undefined') {
      mostrarError('Error', 'No se pudo cargar la librería para generar PDF');
      return;
    }

    const { jsPDF } = window.jspdf;
    const doc = new jsPDF();
    
    // Datos base
    const fechaEmision = new Date().toLocaleDateString('es-CL');
    const horaEmision = new Date().toLocaleTimeString('es-CL');
    const numeroNota = `NV-${Date.now()}`;
    
    // Configuración de fuentes y colores
    const colorPrimario = [59, 130, 246]; // Azul
    const colorSecundario = [107, 114, 128]; // Gris
    
    // ENCABEZADO
    doc.setFillColor(...colorPrimario);
    doc.rect(0, 0, 210, 25, 'F');
    
    doc.setTextColor(255, 255, 255);
    doc.setFontSize(20);
    doc.setFont('helvetica', 'bold');
    doc.text('NOTA DE VENTA', 20, 15);
    
    doc.setFontSize(12);
    doc.setFont('helvetica', 'normal');
    doc.text('Documento de Reserva', 20, 21);
    
    // Número y fecha (derecha)
    doc.text(`No. ${numeroNota}`, 150, 15);
    doc.text(`Fecha: ${fechaEmision}`, 150, 21);
    
    // DATOS DE LA EMPRESA
    let yPos = 35;
    doc.setTextColor(0, 0, 0);
    doc.setFontSize(14);
    doc.setFont('helvetica', 'bold');
    doc.text('DATOS DE LA EMPRESA', 20, yPos);
    
    yPos += 8;
    doc.setFontSize(10);
    doc.setFont('helvetica', 'normal');
    
    const datosEmpresa = [
      `Empresa: ${empresaActual?.nombre || 'MotoExperts Santiago'}`,
      `RUT: ${empresaActual?.rut || '76.123.456-7'}`,
      `Dirección: ${empresaActual?.direccion || 'Av. Providencia 1234, Santiago'}`,
      `Teléfono: +56 2 2345 6789`,
      `Email: ventas@motoexperts.cl`
    ];
    
    datosEmpresa.forEach(dato => {
      doc.text(dato, 20, yPos);
      yPos += 5;
    });
    
    // DATOS DEL CLIENTE
    yPos += 5;
    doc.setFontSize(14);
    doc.setFont('helvetica', 'bold');
    doc.text('DATOS DEL CLIENTE', 20, yPos);
    
    yPos += 8;
    doc.setFontSize(10);
    doc.setFont('helvetica', 'normal');
    
    const datosCliente = [
      `Nombres: ${clienteVenta?.nombres || 'N/A'} ${clienteVenta?.apellidos || ''}`,
      `RUT/Cédula: ${clienteVenta?.cedula || 'N/A'}`,
      `Teléfono: ${clienteVenta?.telefono || 'N/A'}`,
      `Email: ${clienteVenta?.email || 'N/A'}`,
      `Dirección: ${clienteVenta?.direccion || 'N/A'}`
    ];
    
    datosCliente.forEach(dato => {
      doc.text(dato, 20, yPos);
      yPos += 5;
    });
    
    // DATOS DEL VEHÍCULO
    yPos += 5;
    doc.setFontSize(14);
    doc.setFont('helvetica', 'bold');
    doc.text('DATOS DEL VEHÍCULO', 20, yPos);
    
    yPos += 8;
    doc.setFontSize(10);
    doc.setFont('helvetica', 'normal');
    
    const datosVehiculo = [
      `Marca/Modelo: ${motoVenta?.marca || 'N/A'} ${motoVenta?.modelo || ''}`,
      `Año: ${motoVenta?.año || 'N/A'}`,
      `Color: ${motoVenta?.color || 'N/A'}`,
      `Cilindrada: ${motoVenta?.cilindrada || 'N/A'}cc`,
      `Patente: ${motoVenta?.patente || 'N/A'}`,
      `Kilometraje: ${motoVenta?.kms || 0} km`,
      `Estado: ${motoVenta?.condicion || 'Usado'}`
    ];
    
    datosVehiculo.forEach(dato => {
      doc.text(dato, 20, yPos);
      yPos += 5;
    });
    
    // INFORMACIÓN COMERCIAL
    yPos += 5;
    doc.setFillColor(240, 240, 240);
    doc.rect(15, yPos - 3, 180, 35, 'F');
    
    doc.setFontSize(14);
    doc.setFont('helvetica', 'bold');
    doc.text('INFORMACIÓN COMERCIAL', 20, yPos + 5);
    
    yPos += 13;
    doc.setFontSize(12);
    doc.setFont('helvetica', 'normal');
    
    const precioVenta = parseFloat(motoVenta?.precio_publicacion || 0);
    const valorReserva = Math.round(precioVenta * 0.1); // 10% como reserva
    const saldoPendiente = precioVenta - valorReserva;
    
    doc.text(`Precio de Lista: $${precioVenta.toLocaleString('es-CL')}`, 20, yPos);
    doc.text(`Valor de Reserva (10%): $${valorReserva.toLocaleString('es-CL')}`, 20, yPos + 6);
    doc.text(`Saldo Pendiente: $${saldoPendiente.toLocaleString('es-CL')}`, 20, yPos + 12);
    doc.text(`Forma de Pago Reserva: Efectivo/Transferencia`, 20, yPos + 18);
    
    // CONDICIONES
    yPos += 30;
    doc.setFontSize(14);
    doc.setFont('helvetica', 'bold');
    doc.text('CONDICIONES DE LA RESERVA', 20, yPos);
    
    yPos += 8;
    doc.setFontSize(9);
    doc.setFont('helvetica', 'normal');
    
    const condiciones = [
      '• Esta reserva tiene una vigencia de 15 días hábiles desde la fecha de emisión.',
      '• El cliente debe completar la compra dentro del plazo establecido.',
      '• En caso de desistimiento, se reembolsará el 80% del valor de la reserva.',
      '• La moto quedará apartada y no estará disponible para otros clientes.',
      '• Al momento de completar la compra se firmará el contrato definitivo.',
      '• La entrega se realizará previa verificación de documentos y pago total.'
    ];
    
    condiciones.forEach(condicion => {
      doc.text(condicion, 20, yPos);
      yPos += 5;
    });
    
    // FIRMAS
    yPos += 10;
    doc.setFontSize(12);
    doc.setFont('helvetica', 'bold');
    doc.text('VALIDACIÓN', 20, yPos);
    
    yPos += 10;
    doc.setFontSize(10);
    doc.setFont('helvetica', 'normal');
    
    // Cliente
    doc.text('_________________________', 30, yPos + 15);
    doc.text('Firma del Cliente', 30, yPos + 20);
    doc.text(`${clienteVenta?.nombres || 'Cliente'} ${clienteVenta?.apellidos || ''}`, 30, yPos + 25);
    
    // Vendedor
    doc.text('_________________________', 120, yPos + 15);
    doc.text('Firma del Vendedor', 120, yPos + 20);
    doc.text(`${vendedorActual?.nombre || 'Vendedor'}`, 120, yPos + 25);
    
    // Fecha
    doc.text(`Fecha: ${fechaEmision} - ${horaEmision}`, 75, yPos + 35);
    
    // PIE DE PÁGINA
    doc.setFontSize(8);
    doc.setTextColor(...colorSecundario);
    doc.text('Este documento constituye una reserva formal del vehículo descrito.', 20, 280);
    doc.text('Para consultas: ventas@motoexperts.cl | +56 2 2345 6789', 20, 285);
    
    // Guardar datos para el backend
    const datosDocumento = {
      tipo_entidad: 'VENTA',
      id_entidad: null, // Se asignará después de crear la venta
      tipo_documento: 'NOTA_VENTA',
      nombre_archivo: `NotaVenta_${numeroNota}.pdf`,
      contenido_base64: doc.output('datauristring').split(',')[1],
      descripcion: `Nota de venta para reserva - Cliente: ${clienteVenta?.nombres} ${clienteVenta?.apellidos}`,
      es_publico: 0
    };
    
    console.log('✅ Nota de venta generada exitosamente');
    return doc; // RETORNAR el documento PDF
    
  } catch (error) {
    console.error('❌ Error generando Nota de Venta:', error);
    mostrarError('Error', 'No se pudo generar la nota de venta. Por favor, intenta nuevamente.');
    return null; // Retornar null en caso de error
  }
}

// Función para generar Contrato (VENTA DIRECTA)
function generarContratoVentaDirecta() {
  try {
    console.log('📄 Generando Contrato para VENTA DIRECTA');
    
    if (typeof window.jspdf === 'undefined') {
      mostrarError('Error', 'No se pudo cargar la librería para generar PDF');
      return;
    }

    const { jsPDF } = window.jspdf;
    const doc = new jsPDF();
    
    // Datos base
    const fechaContrato = new Date().toLocaleDateString('es-CL');
    const horaContrato = new Date().toLocaleTimeString('es-CL');
    const numeroContrato = `CT-${Date.now()}`;
    
    // Configuración de colores
    const colorPrimario = [16, 185, 129]; // Verde para contrato
    const colorSecundario = [107, 114, 128];
    
    // ENCABEZADO
    doc.setFillColor(...colorPrimario);
    doc.rect(0, 0, 210, 30, 'F');
    
    doc.setTextColor(255, 255, 255);
    doc.setFontSize(18);
    doc.setFont('helvetica', 'bold');
    doc.text('CONTRATO DE COMPRAVENTA', 105, 15, { align: 'center' });
    
    doc.setFontSize(12);
    doc.setFont('helvetica', 'normal');
    doc.text('Vehículo Motorizado', 105, 22, { align: 'center' });
    
    // Información del contrato
    doc.setTextColor(0, 0, 0);
    doc.setFontSize(10);
    doc.text(`Contrato No. ${numeroContrato}`, 20, 40);
    doc.text(`Fecha: ${fechaContrato}`, 20, 45);
    doc.text(`Lugar: Santiago, Chile`, 20, 50);
    
    // PARTES CONTRATANTES
    let yPos = 60;
    doc.setFontSize(14);
    doc.setFont('helvetica', 'bold');
    doc.text('PARTES CONTRATANTES', 20, yPos);
    
    yPos += 10;
    doc.setFontSize(11);
    doc.setFont('helvetica', 'bold');
    doc.text('VENDEDOR:', 20, yPos);
    
    yPos += 5;
    doc.setFontSize(10);
    doc.setFont('helvetica', 'normal');
    
    const datosVendedor = [
      `Razón Social: ${empresaActual?.nombre || 'MotoExperts Santiago'}`,
      `RUT: ${empresaActual?.rut || '76.123.456-7'}`,
      `Representante Legal: Juan Administrador`,
      `Domicilio: ${empresaActual?.direccion || 'Av. Providencia 1234, Santiago'}`,
      `Teléfono: +56 2 2345 6789`
    ];
    
    datosVendedor.forEach(dato => {
      doc.text(dato, 20, yPos);
      yPos += 4;
    });
    
    yPos += 5;
    doc.setFontSize(11);
    doc.setFont('helvetica', 'bold');
    doc.text('COMPRADOR:', 20, yPos);
    
    yPos += 5;
    doc.setFontSize(10);
    doc.setFont('helvetica', 'normal');
    
    const datosComprador = [
      `Nombres: ${clienteVenta?.nombres || 'N/A'} ${clienteVenta?.apellidos || ''}`,
      `RUT/Cédula: ${clienteVenta?.cedula || 'N/A'}`,
      `Domicilio: ${clienteVenta?.direccion || 'N/A'}`,
      `Teléfono: ${clienteVenta?.telefono || 'N/A'}`,
      `Email: ${clienteVenta?.email || 'N/A'}`
    ];
    
    datosComprador.forEach(dato => {
      doc.text(dato, 20, yPos);
      yPos += 4;
    });
    
    // OBJETO DEL CONTRATO
    yPos += 8;
    doc.setFillColor(240, 240, 240);
    doc.rect(15, yPos - 3, 180, 40, 'F');
    
    doc.setFontSize(14);
    doc.setFont('helvetica', 'bold');
    doc.text('OBJETO DEL CONTRATO', 20, yPos + 5);
    
    yPos += 13;
    doc.setFontSize(10);
    doc.setFont('helvetica', 'normal');
    
    const objetoContrato = [
      `Vehículo: ${motoVenta?.marca || 'N/A'} ${motoVenta?.modelo || ''} ${motoVenta?.año || ''}`,
      `Color: ${motoVenta?.color || 'N/A'}`,
      `Número de Motor: ${motoVenta?.numero_motor || 'N/A'}`,
      `Número de Chasis: ${motoVenta?.numero_chasis || 'N/A'}`,
      `Patente: ${motoVenta?.patente || 'N/A'}`,
      `Kilometraje: ${motoVenta?.kms || 0} km`,
      `Estado: ${motoVenta?.condicion || 'Usado'}`
    ];
    
    objetoContrato.forEach(dato => {
      doc.text(dato, 20, yPos);
      yPos += 4;
    });
    
    // CLÁUSULAS COMERCIALES
    yPos += 10;
    doc.setFontSize(14);
    doc.setFont('helvetica', 'bold');
    doc.text('CLÁUSULAS COMERCIALES', 20, yPos);
    
    yPos += 8;
    doc.setFontSize(10);
    doc.setFont('helvetica', 'normal');
    
    const precioVenta = parseFloat(motoVenta?.precio_publicacion || 0);
    
    doc.text(`PRIMERA: El precio total acordado es de $${precioVenta.toLocaleString('es-CL')} (pesos chilenos).`, 20, yPos);
    yPos += 5;
    doc.text(`SEGUNDA: El pago se realiza en forma: Efectivo/Transferencia bancaria.`, 20, yPos);
    yPos += 5;
    doc.text(`TERCERA: La entrega del vehículo se efectúa en fecha ${fechaContrato}.`, 20, yPos);
    yPos += 5;
    doc.text(`CUARTA: El vendedor garantiza el buen estado del vehículo.`, 20, yPos);
    
    // CLÁUSULAS LEGALES
    yPos += 15;
    doc.setFontSize(14);
    doc.setFont('helvetica', 'bold');
    doc.text('CLÁUSULAS LEGALES', 20, yPos);
    
    yPos += 8;
    doc.setFontSize(9);
    doc.setFont('helvetica', 'normal');
    
    const clausulasLegales = [
      'QUINTA: La transferencia de dominio se realizará una vez completado el pago total.',
      'SEXTA: El comprador se hace responsable del vehículo desde la fecha de entrega.',
      'SÉPTIMA: Cualquier conflicto se resolverá según la legislación chilena vigente.',
      'OCTAVA: Este contrato se firma en conformidad por ambas partes.'
    ];
    
    clausulasLegales.forEach(clausula => {
      doc.text(clausula, 20, yPos);
      yPos += 6;
    });
    
    // VALIDACIÓN BIOMÉTRICA
    yPos += 10;
    doc.setFillColor(...colorPrimario);
    doc.rect(15, yPos - 3, 180, 8, 'F');
    
    doc.setTextColor(255, 255, 255);
    doc.setFontSize(12);
    doc.setFont('helvetica', 'bold');
    doc.text('VALIDACIÓN BIOMÉTRICA', 20, yPos + 3);
    
    yPos += 15;
    doc.setTextColor(0, 0, 0);
    doc.setFontSize(10);
    doc.setFont('helvetica', 'normal');
    
    // Firmas
    doc.text('_________________________', 30, yPos);
    doc.text('Firma del Vendedor', 30, yPos + 5);
    doc.text(`${vendedorActual?.nombre || 'Vendedor'}`, 30, yPos + 10);
    
    doc.text('_________________________', 120, yPos);
    doc.text('Firma del Comprador', 120, yPos + 5);
    doc.text(`${clienteVenta?.nombres || ''} ${clienteVenta?.apellidos || ''}`, 120, yPos + 10);
    
    // Huella digital
    yPos += 20;
    doc.setFillColor(200, 200, 200);
    doc.rect(75, yPos, 60, 20, 'F');
    doc.text('HUELLA DIGITAL', 105, yPos + 7, { align: 'center' });
    doc.text('COMPRADOR', 105, yPos + 12, { align: 'center' });
    doc.setFontSize(8);
    doc.text('✓ Validado biométricamente', 105, yPos + 16, { align: 'center' });
    
    // Fecha y hora
    yPos += 25;
    doc.setFontSize(10);
    doc.text(`Fecha: ${fechaContrato} - Hora: ${horaContrato}`, 105, yPos, { align: 'center' });
    doc.text(`Lugar: Santiago, Chile`, 105, yPos + 5, { align: 'center' });
    
    // PIE DE PÁGINA
    doc.setFontSize(8);
    doc.setTextColor(...colorSecundario);
    doc.text('Este contrato ha sido validado con huella digital y constituye un documento legal.', 20, 280);
    
    // Guardar datos para el backend
    const datosDocumento = {
      tipo_entidad: 'VENTA',
      id_entidad: null, // Se asignará después de crear la venta
      tipo_documento: 'CONTRATO',
      nombre_archivo: `Contrato_${numeroContrato}.pdf`,
      contenido_base64: doc.output('datauristring').split(',')[1],
      descripcion: `Contrato de venta directa - Cliente: ${clienteVenta?.nombres} ${clienteVenta?.apellidos}`,
      es_publico: 0
    };
    
    console.log('✅ Contrato generado exitosamente');
    return doc; // RETORNAR el documento PDF
    
  } catch (error) {
    console.error('❌ Error generando Contrato:', error);
    mostrarError('Error', 'No se pudo generar el contrato. Por favor, intenta nuevamente.');
    return null; // Retornar null en caso de error
  }
}

// Función para mostrar visor de Nota de Venta
function mostrarVisorNotaVenta(doc, numeroNota, datosDocumento) {
  const pdfBlob = doc.output('blob');
  const pdfUrl = URL.createObjectURL(pdfBlob);
  
  Swal.fire({
    title: 'Nota de Venta Generada',
    html: `
      <div class="text-left mb-4">
        <p class="text-sm text-gray-600 mb-2">
          <svg class="w-4 h-4 inline mr-1 text-orange-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
          </svg>
          Reserva procesada exitosamente
        </p>
        <p class="text-xs text-gray-500 mb-2">Documento: ${numeroNota}</p>
        <p class="text-xs text-blue-600 mb-4">Estado de la moto actualizado a: <strong>RESERVADA</strong></p>
        <iframe src="${pdfUrl}" width="100%" height="400" style="border: 1px solid #e5e7eb; border-radius: 8px;"></iframe>
      </div>
    `,
    showCancelButton: true,
    confirmButtonText: '📥 Descargar PDF',
    cancelButtonText: '🖨️ Imprimir',
    showDenyButton: true,
    denyButtonText: '✅ Nueva Operación',
    width: '80%',
    customClass: {
      popup: 'rounded-2xl'
    }
  }).then((result) => {
    if (result.isConfirmed) {
      descargarPDF(doc, numeroNota);
    } else if (result.dismiss === Swal.DismissReason.cancel) {
      imprimirPDF(pdfUrl);
    } else if (result.isDenied) {
      // Reiniciar para nueva operación
      reiniciarProceso();
    }
    
    URL.revokeObjectURL(pdfUrl);
  });
}

// Función para mostrar visor de Contrato
function mostrarVisorContrato(doc, numeroContrato, datosDocumento) {
  const pdfBlob = doc.output('blob');
  const pdfUrl = URL.createObjectURL(pdfBlob);
  
  Swal.fire({
    title: 'Contrato de Venta Generado',
    html: `
      <div class="text-left mb-4">
        <p class="text-sm text-gray-600 mb-2">
          <svg class="w-4 h-4 inline mr-1 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
          </svg>
          Venta directa completada exitosamente
        </p>
        <p class="text-xs text-gray-500 mb-2">Contrato: ${numeroContrato}</p>
        <p class="text-xs text-green-600 mb-2">Validación biométrica: <strong>✓ Completada</strong></p>
        <p class="text-xs text-blue-600 mb-4">Estado de la moto actualizado a: <strong>VENDIDA</strong></p>
        <iframe src="${pdfUrl}" width="100%" height="400" style="border: 1px solid #e5e7eb; border-radius: 8px;"></iframe>
      </div>
    `,
    showCancelButton: true,
    confirmButtonText: '📥 Descargar PDF',
    cancelButtonText: '🖨️ Imprimir',
    showDenyButton: true,
    denyButtonText: '✅ Nueva Operación',
    width: '80%',
    customClass: {
      popup: 'rounded-2xl'
    }
  }).then((result) => {
    if (result.isConfirmed) {
      descargarPDF(doc, numeroContrato);
    } else if (result.dismiss === Swal.DismissReason.cancel) {
      imprimirPDF(pdfUrl);
    } else if (result.isDenied) {
      // Reiniciar para nueva operación
      reiniciarProceso();
    }
    
    URL.revokeObjectURL(pdfUrl);
  });
}

// Función para procesar reserva
async function procesarReserva() {
  console.log('🎫 Procesando RESERVA...');
  
  try {
    // Mostrar loading
    Swal.fire({
      title: 'Procesando Reserva...',
      html: `
        <div class="text-center">
          <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-orange-500 mx-auto mb-4"></div>
          <p class="text-sm text-gray-600">Generando Nota de Venta</p>
          <p class="text-xs text-gray-500 mt-2">Reservando moto para el cliente</p>
        </div>
      `,
      allowOutsideClick: false,
      allowEscapeKey: false,
      showConfirmButton: false,
      customClass: {
        popup: 'rounded-2xl'
      }
    });
    
    // Preparar datos del documento
    const datosDocumento = {
      cliente: clienteVenta,
      moto: motoVenta,
      precioVenta: parseFloat(document.getElementById('precioVenta').value) || 0,
      descuento: parseFloat(document.getElementById('descuentoVenta').value) || 0,
      metodoPago: document.getElementById('metodoPago').value || 'EFECTIVO',
      observaciones: document.getElementById('observacionesVenta').value || '',
      tipoDocumento: 'NOTA_VENTA',
      estadoMoto: 'RESERVADA',
      fechaDocumento: new Date(),
      requiereHuella: false
    };
    
    // Calcular total
    datosDocumento.total = datosDocumento.precioVenta - datosDocumento.descuento;
    
    console.log('📋 Datos de la reserva preparados:', datosDocumento);
    
    // 1. PRIMERO: Generar Nota de Venta (documento)
    console.log('📄 Generando Nota de Venta...');
    const doc = generarNotaVentaReserva();
    
    // Verificar que se generó correctamente el documento
    if (!doc) {
      throw new Error('Error al generar la Nota de Venta');
    }
    
    // Generar número de documento
    const numeroNota = `NV-${Date.now()}`;
    
    // 2. SEGUNDO: Guardar documento en BD
    console.log('💾 Guardando documento en base de datos...');
    const motoId = motoVenta.id_moto || motoVenta.id; // Usar id_moto si existe, sino id como fallback
    
    // Preparar datos según estructura de la tabla documentos
    const datosGuardar = {
      tipo_entidad: 'moto',
      id_entidad: motoId,
      tipo_documento: 'NOTA_VENTA',
      nombre_archivo: `NotaVenta_${numeroNota}.pdf`,
      descripcion: `Nota de venta para reserva - Cliente: ${clienteVenta?.nombres} ${clienteVenta?.apellidos} - Moto: ${motoVenta?.marca} ${motoVenta?.modelo}`,
      es_publico: 0,
      archivo_pdf: doc.output('blob')
    };
    
    const documentoGuardado = await guardarDocumento(datosGuardar);
    
    if (!documentoGuardado.success) {
      console.warn('⚠️ No se pudo guardar el documento, pero continuando con el proceso...');
      // No lanzar error para que el proceso continúe
    }
    
    // 3. TERCERO: Actualizar estado de la moto en BD (DESPUÉS del documento)
    console.log('🔄 Actualizando estado de moto a RESERVADA...');
    const resultadoEstado = await actualizarEstadoMoto(motoId, 'RESERVADA');
    if (!resultadoEstado.success) {
      throw new Error('Error al actualizar estado de la moto');
    }
    
    console.log('✅ Reserva procesada exitosamente');
    
    // Preparar datos para el visor
    const datosVisor = {
      cliente: clienteVenta,
      moto: motoVenta,
      precioVenta: parseFloat(document.getElementById('precioVenta').value) || 0,
      descuento: parseFloat(document.getElementById('descuentoVenta').value) || 0,
      metodoPago: document.getElementById('metodoPago').value || 'EFECTIVO',
      observaciones: document.getElementById('observacionesVenta').value || '',
      tipoDocumento: 'NOTA_VENTA',
      estadoMoto: 'RESERVADA',
      fechaDocumento: new Date(),
      requiereHuella: false
    };
    // Calcular total
    datosVisor.total = datosVisor.precioVenta - datosVisor.descuento;
    
    // Cerrar loading y mostrar éxito
    Swal.close();
    
    // Mostrar visor de Nota de Venta
    mostrarVisorNotaVenta(doc, numeroNota, datosVisor);
    
  } catch (error) {
    console.error('❌ Error al procesar reserva:', error);
    
    Swal.fire({
      title: 'Error al procesar reserva',
      text: error.message || 'Ocurrió un error inesperado',
      icon: 'error',
      confirmButtonText: 'Entendido',
      customClass: {
        popup: 'rounded-2xl'
      }
    });
  }
}

// Función auxiliar para actualizar estado de moto
async function actualizarEstadoMoto(idMoto, nuevoEstado) {
  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    
    if (!token) {
      throw new Error('No se encontró token de autenticación');
    }
    
    // Validar que tenemos datos requeridos
    if (!idMoto || !nuevoEstado) {
      throw new Error('ID de moto y nuevo estado son requeridos');
    }
    
    // Validar que el estado sea válido según la base de datos
    const estadosValidos = ['DISPONIBLE', 'RESERVADA', 'VENDIDA', 'EN_MANTENIMIENTO', 'BAJA'];
    if (!estadosValidos.includes(nuevoEstado)) {
      throw new Error(`Estado no válido: ${nuevoEstado}. Debe ser uno de: ${estadosValidos.join(', ')}`);
    }
    
    console.log('🔄 Enviando actualización de estado:', { 
      id_moto: idMoto, 
      estado: nuevoEstado,
      url: 'http://localhost:3001/api/motos/estado'
    });
    
    const response = await fetch('http://localhost:3001/api/motos/estado', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        id: idMoto,  // El backend espera 'id', no 'id_moto'
        estado: nuevoEstado
      })
    });
    
    if (!response.ok) {
      const errorData = await response.json().catch(() => null);
      const errorMessage = errorData?.message || `HTTP error! status: ${response.status}`;
      throw new Error(errorMessage);
    }
    
    const result = await response.json();
    console.log('✅ Estado de moto actualizado:', result);
    return result;
    
  } catch (error) {
    console.error('❌ Error al actualizar estado de moto:', error);
    console.error('❌ Detalles del error:', {
      idMoto,
      nuevoEstado,
      errorMessage: error.message
    });
    return { success: false, error: error.message };
  }
}

// Función auxiliar para guardar documento
async function guardarDocumento(datosDocumento) {
  try {
    const token = localStorage.getItem('authToken') || localStorage.getItem('token');
    
    if (!token) {
      throw new Error('No se encontró token de autenticación');
    }
    
    const formData = new FormData();
    
    // Agregar campos requeridos por el backend
    formData.append('tipo_entidad', datosDocumento.tipo_entidad);
    formData.append('id_entidad', datosDocumento.id_entidad);
    formData.append('tipo_documento', datosDocumento.tipo_documento);
    formData.append('descripcion', datosDocumento.descripcion || '');
    formData.append('es_publico', datosDocumento.es_publico || 0);
    
    // Agregar el PDF como archivo (el backend espera 'documento')
    if (datosDocumento.archivo_pdf) {
      formData.append('documento', datosDocumento.archivo_pdf, datosDocumento.nombre_archivo);
    }
    
    console.log('📤 Enviando documento al servidor...');
    const response = await fetch('http://localhost:3001/api/documentos/upload', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`
      },
      body: formData
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      console.error('❌ Error del servidor:', response.status, errorText);
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    
    const result = await response.json();
    console.log('✅ Documento guardado:', result);
    return result;
    
  } catch (error) {
    console.error('❌ Error al guardar documento:', error);
    return { success: false, error: error.message };
  }
}

// Función para procesar venta directa
async function procesarVentaDirecta() {
  console.log('🏪 INICIO procesarVentaDirecta() - tipoOperacion:', tipoOperacion);
  console.log('🏪 Procesando VENTA DIRECTA...');
  
  try {
    // Mostrar loading
    Swal.fire({
      title: 'Procesando Venta Directa...',
      html: `
        <div class="text-center">
          <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-green-500 mx-auto mb-4"></div>
          <p class="text-sm text-gray-600">Generando Contrato de Venta</p>
          <p class="text-xs text-gray-500 mt-2">Finalizando transacción con validación biométrica</p>
        </div>
      `,
      allowOutsideClick: false,
      allowEscapeKey: false,
      showConfirmButton: false,
      customClass: {
        popup: 'rounded-2xl'
      }
    });
    
    // Preparar datos del documento
    const datosDocumento = {
      cliente: clienteVenta,
      moto: motoVenta,
      precioVenta: parseFloat(document.getElementById('precioVenta').value) || 0,
      descuento: parseFloat(document.getElementById('descuentoVenta').value) || 0,
      metodoPago: document.getElementById('metodoPago').value || 'EFECTIVO',
      observaciones: document.getElementById('observacionesVenta').value || '',
      tipoDocumento: 'CONTRATO',
      estadoMoto: 'VENDIDA',
      fechaDocumento: new Date(),
      requiereHuella: true,
      huellaValidada: huellaAutenticada
    };
    
    // Calcular total
    datosDocumento.total = datosDocumento.precioVenta - datosDocumento.descuento;
    
    console.log('📋 Datos de la venta directa preparados:', datosDocumento);
    
    // 1. PRIMERO: Generar Contrato de Venta (documento)
    console.log('📄 LLAMANDO a generarContratoVentaDirecta()...');
    const doc = generarContratoVentaDirecta();
    console.log('✅ generarContratoVentaDirecta() completado');
    
    // Verificar que se generó correctamente el documento
    if (!doc) {
      throw new Error('Error al generar el Contrato de Venta');
    }
    
    // Generar número de documento
    const numeroContrato = `CV-${Date.now()}`;
    
    // 2. SEGUNDO: Guardar documento en BD
    console.log('💾 Guardando contrato en base de datos...');
    const motoId = motoVenta.id_moto || motoVenta.id; // Usar id_moto si existe, sino id como fallback
    
    // Preparar datos según estructura de la tabla documentos
    const datosGuardar = {
      tipo_entidad: 'moto',
      id_entidad: motoId,
      tipo_documento: 'CONTRATO',
      nombre_archivo: `Contrato_${numeroContrato}.pdf`,
      descripcion: `Contrato de venta directa - Cliente: ${clienteVenta?.nombres} ${clienteVenta?.apellidos} - Moto: ${motoVenta?.marca} ${motoVenta?.modelo}`,
      es_publico: 0,
      archivo_pdf: doc.output('blob')
    };
    
    const documentoGuardado = await guardarDocumento(datosGuardar);
    
    if (!documentoGuardado.success) {
      console.warn('⚠️ No se pudo guardar el documento, pero continuando con el proceso...');
      // No lanzar error para que el proceso continúe
    }
    
    // 3. TERCERO: Actualizar estado de la moto en BD (DESPUÉS del documento)
    console.log('🔄 Actualizando estado de moto a VENDIDA...');
    const resultadoEstado = await actualizarEstadoMoto(motoId, 'VENDIDA');
    if (!resultadoEstado.success) {
      throw new Error('Error al actualizar estado de la moto');
    }
    
    console.log('✅ Venta directa procesada exitosamente');
    
    // Preparar datos para el visor (reutilizamos los datos del documento)
    const datosVisor = {
      cliente: clienteVenta,
      moto: motoVenta,
      precioVenta: datosDocumento.precioVenta,
      descuento: datosDocumento.descuento,
      metodoPago: datosDocumento.metodoPago,
      observaciones: datosDocumento.observaciones,
      tipoDocumento: 'CONTRATO',
      estadoMoto: 'VENDIDA',
      fechaDocumento: datosDocumento.fechaDocumento,
      requiereHuella: true,
      huellaValidada: datosDocumento.huellaValidada,
      total: datosDocumento.total
    };
    
    // Cerrar loading y mostrar éxito
    Swal.close();
    
    // Mostrar visor de Contrato
    mostrarVisorContrato(doc, numeroContrato, datosVisor);
    
  } catch (error) {
    console.error('❌ Error al procesar venta directa:', error);
    
    Swal.fire({
      title: 'Error al procesar venta',
      text: error.message || 'Ocurrió un error inesperado',
      icon: 'error',
      confirmButtonText: 'Entendido',
      customClass: {
        popup: 'rounded-2xl'
      }
    });
  }
}
